<?php

//*************************************************************************************************
//
//                      Required Files
//
//*************************************************************************************************

require_once ('PDFrePROException.class.php');

//*************************************************************************************************
//
//                      Class 'PDFrePRO'
//
//*************************************************************************************************

/**
 * @description PHP class to use the PDFrePRO print service via Internet (requires PHP 7.0+).
 * @package     PDFrePRO
 * @author      RICHTER & POWELEIT GmbH
 * @see         https://www.pdfrepro.de/
 * @version     v3.04
 */
class PDFrePRO
{
    //*********************************************************************************************
    //
    //                  Addresses
    //
    //*********************************************************************************************

    /**
     * The PDFrePRO host, where all requests will be send to.
     */
    private $host = 'https://api.pdfrepro.de';

    //*********************************************************************************************
    //
    //                  Credentials
    //
    //*********************************************************************************************

    /**
     * @var string $apiKey    - The API key, which shall be used for requests.
     */
    private $apiKey    = '';

    /**
     * @var string $sharedKey - The associated shared key to $apiKey.
     */
    private $sharedKey = '';

    //*********************************************************************************************
    //
    //                  URIs
    //
    //*********************************************************************************************

    /**
     * The URI for general requests on placeholders.
     */
    const URI_PLACEHOLDERS              = '/v3/placeholders';

    /**
     * The URI for requests on a specific placeholder.
     */
    const URI_PLACEHOLDERS_ID           = '/v3/placeholders/{id}';

    /**
     * The URI for requests on templates, which are using a specific placeholder.
     */
    const URI_PLACEHOLDERS_ID_TEMPLATES = '/v3/placeholders/{id}/templates';

    /**
     * The URI for general requests on templates.
     */
    const URI_TEMPLATES                 = '/v3/templates';

    /**
     * The URI for requests on a specific template.
     */
    const URI_TEMPLATES_ID              = '/v3/templates/{id}';

    /**
     * The URI for requests on placeholders, which are used by a specific template.
     */
    const URI_TEMPLATES_ID_PLACEHOLDERS = '/v3/templates/{id}/placeholders';

    /**
     * The URI for requests on the WYSIWYG-editor, which is using a specific template.
     */
    const URI_TEMPLATES_ID_EDITOR_URL   = '/v3/templates/{id}/editor-url';

    /**
     * The URI for requests on the PDF of a specific template.
     */
    const URI_TEMPLATES_ID_PDF          = '/v3/templates/{id}/pdf';

    //*********************************************************************************************
    //
    //                  Validations
    //
    //*********************************************************************************************

    /**
     * This array contains all valid response statuses.
     */
    const VALID_STATUSES     = array
                               (
                                   'success',
                                   'error',
                                   'fail'
                               );

    /**
     * This array contains all valid status codes for errors and fails.
     */
    const VALID_STATUS_CODES = array
                               (
                                   400,
                                   401,
                                   404,
                                   405,
                                   406,
                                   408,
                                   409,
                                   411,
                                   500
                               );


    //*********************************************************************************************
    //
    //                  De- / Constructors
    //
    //*********************************************************************************************

    /**
     * PDFrePRO constructor.
     *
     * @param string $apiKey    - The API key, which shall be used for requests.
     * @param string $sharedKey - The associated shared key to $apiKey.
     * @param string $host      - An optional API host, which shall be used for requests.
     *
     * @throws PDFrePROException - If $apiKey or $sharedKey are invalid.
     */
    public function __construct($apiKey, $sharedKey, $host = null)
    {
        // Validate PHP version.
        if (PHP_VERSION_ID < 70000) {
            throw new PDFrePROException('The minimum required PHP version is 7.0.0!', PDFrePROException::CODE_MINIMUM_REQUIRED_PHP_VERSION);
        }

        // Set keys.
        $this->setApiKey($apiKey);
        $this->setSharedKey($sharedKey);

        // Set host.
        if (!empty ($host)) {
            $this->setHost($host);
        }

        // Set internal encoding.
        mb_internal_encoding('UTF-8');
    } // End of constructor.

    /**
     * PDFrePRO destructor.
     */
    public function __destruct()
    {
    } // End of destructor.

    //*********************************************************************************************
    //
    //                  Setter
    //
    //*********************************************************************************************

    /**
     * Sets the API key, which shall be used for requests.
     *
     * @param string $apiKey - The API key, which shall be used for requests.
     *
     * @throws PDFrePROException - If $apiKey is invalid.
     */
    public function setApiKey($apiKey)
    {
        if(!is_string($apiKey) || (mb_strlen($apiKey) !== 20) || !ctype_alnum($apiKey))
        {
            throw new PDFrePROException('Invalid apiKey!');
        }

        $this->apiKey = $apiKey;
    } // End of setter.

    /**
     * Sets the PDFrePRO host, which shall be used for requests.
     *
     * @param string $host - The PDFrePRO host, which shall be used for requests.
     *
     * @throws PDFrePROException - If host is invalid.
     */
    public function setHost($host)
    {
        if (empty ($host)) {
            throw new PDFrePROException('Invalid host!');
        }

        $this->host = $host;
    }

    /**
     * Sets the associated shared key to $apiKey.
     *
     * @param string $sharedKey - The associated shared key to $apiKey.
     *
     * @throws PDFrePROException - If $sharedKey is invalid.
     */
    public function setSharedKey($sharedKey)
    {
        if(!is_string($sharedKey) || (mb_strlen($sharedKey) !== 64) || !ctype_alnum($sharedKey))
        {
            throw new PDFrePROException('Invalid sharedKey!');
        }

        $this->sharedKey = $sharedKey;
    } // End of setter.

    //*********************************************************************************************
    //
    //                  Functions for Placeholders
    //
    //*********************************************************************************************

    /**
     * Creates a new placeholder for your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $name                            - The name of the new placeholder.
     * @param string  $data                            - The data of the new placeholder.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       relative URL to the new placeholder.
     *
     * @return array | string - A relative URL to the new placeholder; or the entire response, if $returnEntireResponse
     *                              is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function createPlaceholder($name,
                                      $data,
                                      $returnEntireResponse = false)
    {
        json_encode($data);
        $this->testJsonError();

        // Send the request.
        $response = $this->sendRequest(self::URI_PLACEHOLDERS,
                                       'POST',
                                       array
                                       (
                                           'name' => $name,
                                           'data' => $data
                                       ));

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    201
                                ));

        // Check, whether the relative URL is available and valid.
        $response = $response['data'];

        if(!isset($response->url) || !is_string($response->url) || (substr_compare($response->url,
                                                                                   self::URI_PLACEHOLDERS . '/',
                                                                                   0,
                                                                                   strlen(self::URI_PLACEHOLDERS) + 1) !== 0))
        {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return $response->url;
    } // End of function 'createPlaceholder'.

    /**
     * Gets a specific placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the placeholder, which shall be requested.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       requested placeholder.
     *
     * @return array - The requested placeholder; or the entire response, if $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getPlaceholder($id,
                                   $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_PLACEHOLDERS_ID));

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response);

        // Check, whether the placeholder is valid.
        $response = $response['data'];

        if(!isset($response->id,
                  $response->name,
                  $response->lastModificationDate,
                  $response->rawData,
                  $response->numberOfReferencedTemplates) || ($response->id !== $id)
                                                          || !is_integer($response->numberOfReferencedTemplates)
                                                          || ($response->numberOfReferencedTemplates < 0))
        {
            throw new PDFrePROException('Response is invalid, due to invalid placeholder!');
        }

        return $response;
    } // End of function 'getPlaceholder'.

    /**
     * Gets all templates of your PDFrePRO account, which are using a specific placeholder.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the placeholder, which is used by the requested
     *                                                       templates.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       requested templates.
     *
     * @return array - The requested templates; or the entire response, if $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getTemplatesByPlaceholder($id,
                                              $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_PLACEHOLDERS_ID_TEMPLATES),
                                       'GET',
                                       array(),
                                       $code);

        // Check, whether the HTTP status code indicates no content.
        if($code === 204)
        {
            // Build a valid response body.
            $response = array
                        (
                            'code'   => 204,
                            'status' => 'success',
                            'data'   => (object)array
                                                (
                                                    'templates' => array()
                                                )
                        );
        }

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    200,
                                    204
                                ));

        // Check, whether templates are available.
        $response = $response['data'];

        if(!isset($response->templates) || !is_array($response->templates))
        {
            throw new PDFrePROException('Response is invalid, due to missing or invalid templates!');
        }

        // Check, whether all templates are valid.
        foreach($response->templates as $template)
        {
            if (!isset ($template->id, $template->name, $template->lastModificationDate)) {
                throw new PDFrePROException('Response is invalid, due to invalid template!');
            }
        }

        return $response->templates;
    } // End of function 'getTemplatesByPlaceholder'.

    /**
     * Gets all placeholders of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       found placeholders.
     *
     * @return array - All placeholders of your PDFrePRO account; or the entire response, if $returnEntireResponse is
     *                     set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getAllPlaceholders($returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(self::URI_PLACEHOLDERS,
                                       'GET',
                                       array(),
                                       $code);

        // Check, whether the HTTP status code indicates no content.
        if($code === 204)
        {
            // Build a valid response body.
            $response = array
                        (
                            'code'   => 204,
                            'status' => 'success',
                            'data'   => (object)array
                                                (
                                                    'placeholders' => array()
                                                )
                        );
        }

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    200,
                                    204
                                ));

        // Check, whether placeholders are available.
        $response = $response['data'];

        if(!isset($response->placeholders) || !is_array($response->placeholders))
        {
            throw new PDFrePROException('Response is invalid, due to missing or invalid placeholders!');
        }

        // Check, whether all placeholders are valid.
        foreach($response->placeholders as $placeholder)
        {
            if(!isset($placeholder->id,
                      $placeholder->name,
                      $placeholder->lastModificationDate,
                      $placeholder->numberOfReferencedTemplates) || !is_integer($placeholder->numberOfReferencedTemplates)
                                                                 || ($placeholder->numberOfReferencedTemplates < 0))
            {
                throw new PDFrePROException('Response is invalid, due to invalid placeholder!');
            }
        }

        return $response->placeholders;
    } // End of function 'getAllPlaceholders'.

    /**
     * Updates an existing placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the placeholder, which shall be updated.
     * @param string  $name                 (optional) - The new name of the placeholder.
     * @param string  $data                 (optional) - The new data of the placeholder.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just whether
     *                                                       the placeholder got updated.
     *
     * @return array | boolean - Whether the placeholder got updated ('true', 'false'); or the entire response, if
     *                               $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function updatePlaceholder($id,
                                      $name                 = '',
                                      $data                 = '',
                                      $returnEntireResponse = false)
    {
        // Prepare the request.
        $rData = array();

        if(is_string($name) && ($name !== ''))
        {
            $rData['name'] = $name;
        }
        if(is_string($data) && ($data !== ''))
        {
            json_decode($data);
            $this->testJsonError();

            $rData['data'] = $data;
        }

        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_PLACEHOLDERS_ID),
                                       'PUT',
                                       $rData);

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response);

        // Check, whether the relative URL is available and valid.
        if(!isset($response['data']->url)  || (str_replace('{id}',
                                                           $id,
                                                           self::URI_PLACEHOLDERS_ID) !== $response['data']->url))
        {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return ($response['status'] === 'success');
    } // End of function 'updatePlaceholder'.

    /**
     * Copies an existing placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the placeholder, which shall be copied.
     * @param string  $name                 (optional) - The name of the copied placeholder.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       relative URL to the copied placeholder.
     *
     * @return array | string - A relative URL to the copied placeholder; or the entire response, if
     *                              $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function copyPlaceholder($id,
                                    $name                 = '',
                                    $returnEntireResponse = false)
    {
        // Prepare the request.
        $data = array();

        if(is_string($name) && ($name !== ''))
        {
            $data['name'] = $name;
        }

        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_PLACEHOLDERS_ID),
                                       'POST',
                                       $data);

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    201
                                ));

        // Check, whether the relative URL is available and valid.
        $response = $response['data'];

        if(!isset($response->url) || !is_string($response->url) || (substr_compare($response->url,
                                                                                   self::URI_PLACEHOLDERS . '/',
                                                                                   0,
                                                                                   strlen(self::URI_PLACEHOLDERS) + 1) !== 0))
        {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return $response->url;
    } // End of function 'copyPlaceholder'.

    /**
     * Deletes an existing placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the placeholder, which shall be deleted.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just whether
     *                                                       the placeholder got deleted.
     *
     * @return array | boolean - Whether the placeholder got deleted ('true', 'false'); or the entire response, if
     *                               $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function deletePlaceholder($id,
                                      $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_PLACEHOLDERS_ID),
                                       'DELETE',
                                       array(),
                                       $code);

        // Check, whether the HTTP status code indicates no content.
        if($code === 204)
        {
            // Build a valid response body.
            $response = array
                        (
                            'code'   => 204,
                            'status' => 'success',
                            'data'   => new stdClass()
                        );
        }

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    204
                                ));

        // Return the result.
        return ($response['status'] === 'success');
    } // End of function 'deletePlaceholder'.

    //*********************************************************************************************
    //
    //                  Functions for Templates
    //
    //*********************************************************************************************

    /**
     * Creates a new template for your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $name                            - The name of the new template.
     * @param string  $description          (optional) - The description of the new template.
     * @param array   $placeholderIds       (optional) - The IDs of all placeholders, which shall be used by the new
     *                                                       template.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       relative URL to the new template.
     *
     * @return array | string - A relative URL to the new template; or the entire response, if $returnEntireResponse is
     *                              set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function createTemplate(      $name,
                                         $description          = '',
                                   array $placeholderIds       = [],
                                         $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(self::URI_TEMPLATES,
                                       'POST',
                                       array
                                       (
                                           'name'           => $name,
                                           'description'    => $description,
                                           'placeholderIds' => $placeholderIds
                                       ));

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    201
                                ));

        // Check, whether the relative URL is available and valid.
        $response = $response['data'];

        if(!isset($response->url) || !is_string($response->url) || (substr_compare($response->url,
                                                                                   self::URI_TEMPLATES . '/',
                                                                                   0,
                                                                                   strlen(self::URI_TEMPLATES) + 1) !== 0))
        {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return $response->url;
    } // End of function 'createTemplate'.

    /**
     * Gets a specific template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the template, which shall be requested.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       requested template.
     *
     * @return array - The requested template; or the entire response, if $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getTemplate($id,
                                $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_TEMPLATES_ID));

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response);

        // Check, whether the template is valid.
        $response = $response['data'];

        if (!isset ($response->id, $response->name, $response->lastModificationDate) || ($response->id !== $id)) {
            throw new PDFrePROException('Response is invalid, due to invalid template!');
        }

        return $response;
    } // End of function 'getTemplate'.

    /**
     * Gets all placeholders of your PDFrePRO account, which are used by a specific template.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the template, which uses the requested placeholders.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       requested placeholders.
     *
     * @return array - The requested placeholders; or the entire response, if $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getPlaceholdersByTemplate($id,
                                              $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_TEMPLATES_ID_PLACEHOLDERS),
                                       'GET',
                                       array(),
                                       $code);

        // Check, whether the HTTP status code indicates no content.
        if($code === 204)
        {
            // Build a valid response body.
            $response = array
                        (
                            'code'   => 204,
                            'status' => 'success',
                            'data'   => (object)array
                                                (
                                                    'placeholders' => array()
                                                )
                        );
        }

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    200,
                                    204
                                ));

        // Check, whether placeholder are available.
        $response = $response['data'];

        if(!isset($response->placeholders) || !is_array($response->placeholders))
        {
            throw new PDFrePROException('Response is invalid, due to missing or invalid placeholder IDs!');
        }

        // Check, whether all placeholder are valid.
        foreach($response->placeholders as $placeholder)
        {
            if(!isset($placeholder->id,
                      $placeholder->name,
                      $placeholder->lastModificationDate,
                      $placeholder->numberOfReferencedTemplates) || !is_integer($placeholder->numberOfReferencedTemplates)
                                                                 || ($placeholder->numberOfReferencedTemplates < 0))
            {
                throw new PDFrePROException('Response is invalid, due to invalid placeholder!');
            }
        }

        return $response->placeholders;
    } // End of function 'getPlaceholdersByTemplate'.

    /**
     * Gets all templates of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       found templates.
     *
     * @return array - All templates of your PDFrePRO account; or the entire response, if $returnEntireResponse is set
     *                     to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getAllTemplates($returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(self::URI_TEMPLATES,
                                       'GET',
                                       array(),
                                       $code);

        // Check, whether the HTTP status code indicates no content.
        if($code === 204)
        {
            // Build a valid response body.
            $response = array
                        (
                            'code'   => 204,
                            'status' => 'success',
                            'data'   => (object)array
                                                (
                                                    'templates' => array()
                                                )
                        );
        }

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    200,
                                    204
                                ));

        // Check, whether templates are available.
        $response = $response['data'];

        if(!isset($response->templates) || !is_array($response->templates))
        {
            throw new PDFrePROException('Response is invalid, due to missing or invalid templates!');
        }

        // Check, whether all templates are valid.
        foreach($response->templates as $template)
        {
            if (!isset ($template->id, $template->name, $template->lastModificationDate)) {
                throw new PDFrePROException('Response is invalid, due to invalid template!');
            }
        }

        return $response->templates;
    } // End of function 'getAllTemplates'.

    /**
     * Gets an URL, which opens the WYSIWYG editor, to edit an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * IMPORTANT NOTE:
     * Before you start using our PDFrePRO-API, please make sure you have everything
     * configured correctly in your account in our Portal (https://portal.pdfrepro.de).
     * When you are intending to integrate our PDFrePRO-WYSIWYG-Editor into your own
     * application you need to set up a success- and abort-url as follow-up urls to
     * let the Editor know where to redirect to when you want to leave the editor.
     *
     * @param string  $id                              - The ID of the template, which shall be opened in the WYSIWYG
     *                                                       editor.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the URL
     *                                                       to the WYSIWYG editor.
     *
     * @return array | string - An URL to open the WYSIWYG editor; or the entire response, if $returnEntireResponse is
     *                              set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getEditorUrl($id, $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}', $id, self::URI_TEMPLATES_ID_EDITOR_URL));

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true) {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response);

        // Check, whether the URL is available and valid.
        $response = $response['data'];

        if(!isset($response->url) || !is_string($response->url)) {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return $response->url;
    } // End of function 'getEditorUrl'.

    /**
     * Gets a Base64-encoded PDF of an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the template, which shall be printed as PDF.
     * @param array   $data                 (optional) - The data for the placeholders, which are used by the template.
     *                                                   Example: array
     *                                                            (
     *                                                                <placeholderName> => array
     *                                                                                     (
     *                                                                                         <placeholderDataName> => <placeholderDataValue>
     *                                                                                     )
     *                                                            );
     * @param string  $language             (optional) - Any language, you have defined in settings of your API key.
     *                                                   Example: 'en'
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the PDF.
     *
     * @return array | string - A Base64-encoded PDF; or the entire response, if $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function getPDF(      $id,
                           array $data                 = [],
                                 $language             = '',
                                 $returnEntireResponse = false)
    {
        $dataString = json_encode($data);
        $this->testJsonError();

        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_TEMPLATES_ID_PDF),
                                       'POST',
                                       array
                                       (
                                           'data'     => $dataString,
                                           'language' => $language
                                       ));

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    201, 429
                                ));

        // Check, whether the PDF is available and valid.
        $response = $response['data'];

        if(!isset($response->pdf) || !is_string($response->pdf))
        {
            throw new PDFrePROException('Response is invalid, due to invalid PDF!');
        }

        return $response->pdf;
    } // End of function 'getPDF'.

    /**
     * Updates an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the template, which shall be updated.
     * @param string  $name                 (optional) - The new name of the template.
     * @param string  $description          (optional) - The new description of the template.
     * @param array   $placeholderIds       (optional) - The IDs of all placeholders, which shall be used by the
     *                                                       template. (Removes all existing usages of placeholders by
     *                                                       the template.)
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just whether
     *                                                       the template got updated.
     *
     * @return array | boolean - Whether the template got updated ('true', 'false'); or the entire response, if
     *                               $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function updateTemplate(      $id,
                                         $name                = '',
                                         $description          = '',
                                   array $placeholderIds       = null,
                                         $returnEntireResponse = false)
    {
        // Prepare the request.
        $data = array();

        if(is_string($name) && ($name !== ''))
        {
            $data['name'] = $name;
        }
        if(is_string($description) && ($description !== ''))
        {
            $data['description'] = $description;
        }
        if($placeholderIds !== null)
        {
            $data['placeholderIds'] = $placeholderIds;
        }

        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_TEMPLATES_ID),
                                       'PUT',
                                       $data);

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response);

        // Check, whether the relative URL is available and valid.
        if(!isset($response['data']->url)  || (str_replace('{id}',
                                                           $id,
                                                           self::URI_TEMPLATES_ID) !== $response['data']->url))
        {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return ($response['status'] === 'success');
    } // End of function 'updateTemplate'.

    /**
     * Copies an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the template, which shall be copied.
     * @param string  $name                 (optional) - The name of the copied template.
     * @param string  $description          (optional) - The description of the copied template.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just the
     *                                                       relative URL to the copied template.
     *
     * @return array | string - A relative URL to the copied template; or the entire response, if $returnEntireResponse
     *                              is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function copyTemplate($id,
                                 $name                 = '',
                                 $description          = '',
                                 $returnEntireResponse = false)
    {
        // Prepare the request.
        $data = array();

        if(is_string($name) && ($name !== ''))
        {
            $data['name'] = $name;
        }
        if(is_string($description) && ($description !== ''))
        {
            $data['description'] = $description;
        }

        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_TEMPLATES_ID),
                                       'POST',
                                       $data);

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    201
                                ));

        // Check, whether the relative URL is available and valid.
        $response = $response['data'];

        if(!isset($response->url) || !is_string($response->url) || (substr_compare($response->url,
                                                                                   self::URI_TEMPLATES . '/',
                                                                                   0,
                                                                                   strlen(self::URI_TEMPLATES) + 1) !== 0))
        {
            throw new PDFrePROException('Response is invalid, due to invalid URL!');
        }

        return $response->url;
    } // End of function 'copyTemplate'.

    /**
     * Deletes an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param string  $id                              - The ID of the template, which shall be deleted.
     * @param boolean $returnEntireResponse (optional) - Whether the entire response shall be returned, or just whether
     *                                                       the template got deleted.
     *
     * @return array | boolean - Whether the template got deleted ('true', 'false'); or the entire response, if
     *                               $returnEntireResponse is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains
     *                                  an error.
     */
    public function deleteTemplate($id,
                                   $returnEntireResponse = false)
    {
        // Send the request.
        $response = $this->sendRequest(str_replace('{id}',
                                                   $id,
                                                   self::URI_TEMPLATES_ID),
                                       'DELETE',
                                       array(),
                                       $code);

        // Check, whether the HTTP status code indicates no content.
        if($code === 204)
        {
            // Build a valid response body.
            $response = array
                        (
                            'code'   => 204,
                            'status' => 'success',
                            'data'   => new stdClass()
                        );
        }

        // Check, whether the entire response shall be returned.
        if($returnEntireResponse === true)
        {
            return $response;
        }

        // Validate the response.
        $this->validateResponse($response,
                                array
                                (
                                    204
                                ));

        // Return the result.
        return ($response['status'] === 'success');
    } // End of function 'deleteTemplate'.

    //*********************************************************************************************
    //
    //                  Response Functions
    //
    //*********************************************************************************************

    /**
     * Validates a response, which were returned, due to a request to the PDFrePRO host.
     *
     * @param array $response              - The response, which shall be validated.
     * @param array $validCodes (optional) - All valid HTTP status codes, which are expected for a success of this
     *                                           response.
     *                                       If not set, only HTTP status code 200 - OK is expected for a success.
     *
     * @throws PDFrePROException - If the response is not valid or contains an error.
     */
    protected function validateResponse(array $response,
                                        array $validCodes = [200])
    {
        // Check, whether code, status and data are available.
        if(!isset($response['code'],
                  $response['status'],
                  $response['data']))
        {
            throw new PDFrePROException('Response is invalid, due to missing code, status or data!');
        }

        // Check, whether code and status are valid.
        if(!in_array($response['code'],
                     array_merge($validCodes,
                                 self::VALID_STATUS_CODES),
                     true) || !in_array($response['status'],
                                        self::VALID_STATUSES,
                                        true))
        {
            throw new PDFrePROException('Response is invalid, due to invalid code or status!');
        }

        // Check, whether the response contains an error.
        if($response['status'] === 'success')
        {
            // Check, whether data is an object.
            if(!is_object($response['data']))
            {
                throw new PDFrePROException('Response is invalid, due to invalid data!');
            }
        }
        else
        {
            // Check, whether message is available.
            if(!isset($response['message']))
            {
                throw new PDFrePROException('Response is invalid, due to missing message!');
            }

            // Check, whether data and message are strings.
            if(!is_string($response['data']) || !is_string($response['message']))
            {
                throw new PDFrePROException('Response is invalid, due to invalid data or message!');
            }

            // Throw proper exception.
            throw new PDFrePROException($response['data'] . ': ' . $response['message'],
                                 $response['code']);
        }
    } // End of function 'validateResponse'.

    //*********************************************************************************************
    //
    //                  Request Functions
    //
    //*********************************************************************************************

    /**
     * Sends a request.
     *
     * @param string  $resource             - The resource, which shall be requested.
     * @param string  $method   (optional)  - The HTTP method for the request.
     * @param array   $data     (optional)  - The Data, which shall be sent with the request.
     * @param integer $httpCode (reference,
     *                           out,
     *                           optional)  - This parameter will set to hold the HTTP status code of the cURL request.
     *
     * @return array - The result of the request.
     *
     * @throws PDFrePROException - If the request could not be send, properly.
     */
    protected function sendRequest(      $resource,
                                         $method    = 'GET',
                                   array $data      = [],
                                         &$httpCode = 0)
    {
        return $this->runCurl($this->initCurl($resource,
                                              $method,
                                              $data),
                              $httpCode);
    } // End of function 'sendRequest'.

    /**
     * Initializes a new cURL session.
     *
     * @param string $resource - The resource, which shall be requested.
     * @param string $method   - The HTTP method for the request.
     * @param array  $data     - The Data, which shall be send with the request.
     *
     * @return resource - A handle to a cURL session.
     *
     * @throws PDFrePROException - If the cURL session could not be initialized, properly.
     */
    private function initCurl(      $resource,
                                    $method,
                              array $data)
    {
        // Initialize cURL session.
        $curl = curl_init($this->host . $resource);

        if($curl === false)
        {
            throw new PDFrePROException('cURL session could not be initialized!');
        }

        // Set general cURL options.
        $result = curl_setopt_array($curl,
                                    array
                                    (
                                        CURLOPT_CUSTOMREQUEST  => $method,
                                        CURLOPT_RETURNTRANSFER => true,
                                        CURLOPT_SSL_VERIFYHOST => 2,
                                        CURLOPT_SSL_VERIFYPEER => true
                                    ));

        if(!$result)
        {
            throw new PDFrePROException(curl_error($curl),
                                         curl_errno($curl));
        }

        // Initialize headers and header array.
        $accept        = 'application/json;charset=utf-8';
        $contentType   = '';
        $contentLength = '';
        $headers       = array
                         (
                             'Accept: ' . $accept
                         );

        // Set cURL options, depending on request method.
        if(($method === 'POST') || ($method === 'PUT'))
        {
            // Convert data to string.
            $dataString = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRESERVE_ZERO_FRACTION);

            // Add data to cURL session.
            $result = curl_setopt($curl,
                                  CURLOPT_POSTFIELDS,
                                  $dataString);

            if(!$result)
            {
                throw new PDFrePROException(curl_error($curl),
                                             curl_errno($curl));
            }

            // Add content headers.
            $contentType   = 'application/json;charset=utf-8';
            $contentLength = strlen($dataString);

            array_push($headers,
                       'Content-Type: '   . $contentType,
                       'Content-Length: ' . $contentLength);
        }

        // Add date header.
        $timezone = date_default_timezone_get();

        date_default_timezone_set('GMT');

        $date = date('D, d M Y G:i:s',
                     time()) . ' GMT';

        date_default_timezone_set($timezone);
        array_push($headers,
                   'Date: ' . $date);

        // Extract host and port of host-URL.
        $host = $this->extractHost();

        // Add authorization header.
        $hash  = $this->apiKey                                . '\n';
        $hash .= $method                                      . '\n';
        $hash .= $resource                                    . '\n';
        $hash .= $host                                        . '\n';
        $hash .= (isset($dataString) ? md5($dataString) : '') . '\n';
        $hash .= $accept                                      . '\n';
        $hash .= $contentType                                 . '\n';
        $hash .= $contentLength                               . '\n';
        $hash .= $date                                        . '\n';

        $hash  = hash_hmac('sha256',
                           $hash,
                           $this->sharedKey,
                           false);

        array_push($headers,
                   'Authorization: SharedKey ' . $this->apiKey . ':' . $hash);

        // Set headers.
        $result = curl_setopt($curl,
                              CURLOPT_HTTPHEADER,
                              $headers);

        if(!$result)
        {
            throw new PDFrePROException(curl_error($curl),
                                         curl_errno($curl));
        }

        // Everything went well.
        return $curl;
    } // End of function 'initCurl'.

    /**
     * Executes a cURL session.
     *
     * @param resource $curl                 - A handle to a cURL session.
     * @param integer  $httpCode (reference,
     *                            out)       - This parameter will set to hold the HTTP status code of the cURL request.
     *
     * @return array - The result of the cURL session.
     *
     * @throws PDFrePROException - If the cURL session could not be executed, properly.
     */
    private function runCurl($curl,
                             &$httpCode)
    {
        // Execute the cURL session.
        $result = curl_exec($curl);

        if($result === false)
        {
            throw new PDFrePROException(curl_error($curl),
                                         curl_errno($curl));
        }

        // Set HTTP status code of the cURL request.
        $httpCode = curl_getinfo($curl,
                                 CURLINFO_HTTP_CODE);

        if($result === false)
        {
            throw new PDFrePROException(curl_error($curl),
                                         curl_errno($curl));
        }

        // Close the cURL session and return the result.
        curl_close($curl);

        return (array)json_decode($result);
    } // End of function 'runCurl'.

    //*********************************************************************************************
    //
    //                  Helper Functions
    //
    //*********************************************************************************************

    /**
     * Extracts the host and port out of the stored host address of this class instance and returns it back as
     * concatenated string.
     *
     * @return string - The extracted and concatenated host and port of the stored host address of this class instance.
     */
    private function extractHost()
    {
        $host = parse_url($this->host,
                          PHP_URL_HOST);
        $port = parse_url($this->host,
                          PHP_URL_PORT);

        if(!empty($port))
        {
            $host .= ":" . $port;
        }

        return $host;
    } // End of function 'extractHost'.

    /**
     * Throw exception if an error occurred during the last JSON conversion
     *
     * @throws Exception
     */
    private function testJsonError()
    {
        $lastError = json_last_error();

        if($lastError !== JSON_ERROR_NONE)
        {
            switch ($lastError)
            {
                case JSON_ERROR_DEPTH:
                    $errorMessage = 'JSON-Error - The maximum stack depth has been exceeded';
                    break;

                case JSON_ERROR_STATE_MISMATCH:
                    $errorMessage = 'JSON-Error - Invalid or malformed JSON';
                    break;

                case JSON_ERROR_CTRL_CHAR:
                    $errorMessage = 'JSON-Error - Control character error, possibly incorrectly encoded';
                    break;

                case JSON_ERROR_SYNTAX:
                    $errorMessage = 'JSON-Error - Syntax error, malformed JSON';
                    break;

                case JSON_ERROR_UTF8:
                    $errorMessage = 'JSON-Error - Malformed UTF-8 characters, possibly incorrectly encoded';
                    break;

                case JSON_ERROR_RECURSION:
                    $errorMessage = 'JSON-Error - One or more recursive references in the value to be encoded';
                    break;

                case JSON_ERROR_INF_OR_NAN:
                    $errorMessage = 'JSON-Error - One or more NAN or INF values in the value to be encoded';
                    break;

                case JSON_ERROR_UNSUPPORTED_TYPE:
                    $errorMessage = 'JSON-Error - A value of a type that cannot be encoded was given';
                    break;

                case JSON_ERROR_INVALID_PROPERTY_NAME:
                    $errorMessage = 'JSON-Error - A property name that cannot be encoded was given';
                    break;

                case JSON_ERROR_UTF16:
                    $errorMessage = 'JSON-Error - Malformed UTF-16 characters, possibly incorrectly encoded';
                    break;

                default:
                    $errorMessage = 'Content is not JSON encoded!';
            }

            throw new Exception($errorMessage);
        }
    } // End of function 'testJsonError'.
}
