
//*************************************************************************************************
//
//                      Package 'json'
//
//*************************************************************************************************

package json;

//*************************************************************************************************
//
//                      Imports
//
//*************************************************************************************************

// Package 'java.io'.
import java.io.Serializable;

// Package 'java.lang'.
import java.lang.Cloneable;

// Package 'javax.lang.model.type'.
import javax.lang.model.type.NullType;

//*************************************************************************************************
//
//                      Class 'JsonValue'
//
//*************************************************************************************************

/**
 * This class represents a JSON value.
 */
public final class JsonValue implements Cloneable,
										Serializable
{

	//*********************************************************************************************
	//
	//                  Serialization Attributes
	//
	//*********************************************************************************************
	
	/**
	 * The unique serial version ID of this class.
	 * 
	 * Note:
	 * Don't forget to increment the ID, after you made changes to this class.
	 */
	private static final long serialVersionUID = 0L;
	
	//*********************************************************************************************
	//
	//                  JSON Attributes
	//
	//*********************************************************************************************
	
	/**
	 * This attribute stores the value of this instance.
	 */
	private Object value = null;
	
	//*********************************************************************************************
	//
	//                  Constructors
	//
	//*********************************************************************************************
	
	/**
	 * JsonValue constructor for 'null'.
	 */
	public JsonValue()
	{
	} // End of constructor.

	/**
	 * JsonValue constructor for 'null'.
	 * 
	 * @param NullType value - 'null'.
	 */
	public JsonValue(NullType value)
	{
		this.value = value;
	} // End of constructor.
	
	/**
	 * JsonValue constructor for booleans.
	 * 
	 * @param Boolean value - A boolean.
	 */
	public JsonValue(Boolean value)
	{
		this.value = value;
	} // End of constructor.

	/**
	 * JsonValue constructor for numbers.
	 * 
	 * @param Number value - A number.
	 */
	public JsonValue(Number value)
	{
		this.value = value;
	} // End of constructor.
	
	/**
	 * JsonValue constructor for strings.
	 * 
	 * @param String value - A string.
	 */
	public JsonValue(String value)
	{
		this.value = value;
	} // End of constructor.
	
	/**
	 * JsonValue constructor for JSON arrays.
	 * 
	 * @param JsonArray value - A JSON array.
	 */
	public JsonValue(JsonArray value)
	{
		this.value = value;
	} // End of constructor.
	
	/**
	 * JsonValue constructor for JSON objects.
	 * 
	 * @param JsonObject value - A JSON object.
	 */
	public JsonValue(JsonObject value)
	{
		this.value = value;
	} // End of constructor.
	
	/**
	 * JsonValue copy constructor.
	 * 
	 * @param JsonValue value - A JSON value.
	 */
	public JsonValue(JsonValue value)
	{
		this.value = value.getValue();
	} // End of constructor.
	
	//*********************************************************************************************
	//
	//                  Getters
	//
	//*********************************************************************************************
	
	/**
	 * Gets the value type of this instance.
	 * 
	 * @return Class<?> - The value type of this instance.
	 */
	public Class<?> getType()
	{
		return this.value.getClass();
	} // End of getter.
	
	/**
	 * Gets the value of this instance (without cast).
	 * 
	 * @return Object - The value of this instance (without cast).
	 */
	public Object getValue()
	{
		return this.value;
	} // End of getter.

	/**
	 * Gets the null value of this instance.
	 * 
	 * @return NullType - The null value of this instance.
	 * 
	 * @throws ClassCastException - If the value of this instance is not 'null'.
	 */
	public NullType getNullValue() throws ClassCastException
	{
		if(!this.isNull())
		{
			throw new ClassCastException(this.value.getClass().getName() + " cannot be cast to " + null);
		}
		
		return (NullType)this.value;
	} // End of getter.
	
	/**
	 * Gets the boolean value of this instance.
	 * 
	 * @return Boolean - The boolean value of this instance.
	 * 
	 * @throws ClassCastException - If the value of this instance is not a boolean.
	 */
	public Boolean getBooleanValue() throws ClassCastException
	{
		if(!this.isBoolean())
		{
			throw new ClassCastException((this.isNull() ? null : this.value.getClass().getName()) + " cannot be cast to " + Boolean.class.getName());
		}
		
		return (Boolean)this.value;
	} // End of getter.
	
	/**
	 * Gets the number value of this instance.
	 * 
	 * @return Number - The number value of this instance.
	 * 
	 * @throws ClassCastException - If the value of this instance is not a number.
	 */
	public Number getNumberValue() throws ClassCastException
	{
		if(!this.isNumber())
		{
			throw new ClassCastException((this.isNull() ? null : this.value.getClass().getName()) + " cannot be cast to " + Number.class.getName());
		}
		
		return (Number)this.value;
	} // End of getter.
	
	/**
	 * Gets the string value of this instance.
	 * 
	 * @return String - The string value of this instance.
	 * 
	 * @throws ClassCastException - If the value of this instance is not a string.
	 */
	public String getStringValue() throws ClassCastException
	{
		if(!this.isString())
		{
			throw new ClassCastException((this.isNull() ? null : this.value.getClass().getName()) + " cannot be cast to " + String.class.getName());
		}
		
		return (String)this.value;
	} // End of getter.

	/**
	 * Gets the JSON array value of this instance.
	 * 
	 * @return JsonArray - The JSON array value of this instance.
	 * 
	 * @throws ClassCastException - If the value of this instance is not a JSON array.
	 */
	public JsonArray getJsonArrayValue() throws ClassCastException
	{
		if(!this.isJsonArray())
		{
			throw new ClassCastException((this.isNull() ? null : this.value.getClass().getName()) + " cannot be cast to " + JsonArray.class.getName());
		}
		
		return (JsonArray)this.value;
	} // End of getter.

	/**
	 * Gets the JSON object value of this instance.
	 * 
	 * @return JsonObject - The JSON object value of this instance.
	 * 
	 * @throws ClassCastException - If the value of this instance is not a JSON object.
	 */
	public JsonObject getJsonObjectValue() throws ClassCastException
	{
		if(!this.isJsonObject())
		{
			throw new ClassCastException((this.isNull() ? null : this.value.getClass().getName()) + " cannot be cast to " + JsonObject.class.getName());
		}
		
		return (JsonObject)this.value;
	} // End of getter.
	
	//*********************************************************************************************
	//
	//                  Setter
	//
	//*********************************************************************************************
	
	/**
	 * Sets the value of this instance to 'null'.
	 */
	public void setValue()
	{
		this.value = null;
	} // End of setter.
	
	/**
	 * Sets the value of this instance to 'null'.
	 * 
	 * @param NullType value - The new value of this instance.
	 */
	public void setValue(NullType value)
	{
		this.value = value;
	} // End of setter.
	
	/**
	 * Sets the value of this instance to a boolean.
	 * 
	 * @param Boolean value - The new value of this instance.
	 */
	public void setValue(Boolean value)
	{
		this.value = value;
	} // End of setter.
	
	/**
	 * Sets the value of this instance to a number.
	 * 
	 * @param Number value - The new value of this instance.
	 */
	public void setValue(Number value)
	{
		this.value = value;
	} // End of setter.
	
	/**
	 * Sets the value of this instance to a string.
	 * 
	 * @param String value - The new value of this instance.
	 */
	public void setValue(String value)
	{
		this.value = value;
	} // End of setter.
	
	/**
	 * Sets the value of this instance to a JSON array.
	 * 
	 * @param JsonArray value - The new value of this instance.
	 */
	public void setValue(JsonArray value)
	{
		this.value = value;
	} // End of setter.
	
	/**
	 * Sets the value of this instance to a JSON object.
	 * 
	 * @param JsonObject value - The new value of this instance.
	 */
	public void setValue(JsonObject value)
	{
		this.value = value;
	} // End of setter.
	
	//*********************************************************************************************
	//
	//                  Status Methods
	//
	//*********************************************************************************************
	
	/**
	 * Whether the value of this instance is 'null'.
	 * 
	 * @return Boolean - 'true', if the value of this instance is 'null'. 'false', otherwise.
	 */
	public Boolean isNull()
	{
		return new Boolean(this.value == null);
	} // End of status method.

	/**
	 * Whether the value of this instance is a boolean.
	 * 
	 * @return Boolean - 'true', if the value of this instance is a boolean. 'false', otherwise.
	 */
	public Boolean isBoolean()
	{
		return new Boolean(this.value instanceof Boolean);
	} // End of status method.

	/**
	 * Whether the value of this instance is a number.
	 * 
	 * @return Boolean - 'true', if the value of this instance is a number. 'false', otherwise.
	 */
	public Boolean isNumber()
	{
		return new Boolean(this.value instanceof Number);
	} // End of status method.

	/**
	 * Whether the value of this instance is a string.
	 * 
	 * @return Boolean - 'true', if the value of this instance is a string. 'false', otherwise.
	 */
	public Boolean isString()
	{
		return new Boolean(this.value instanceof String);
	} // End of status method.

	/**
	 * Whether the value of this instance is a JSON array.
	 * 
	 * @return Boolean - 'true', if the value of this instance is JSON array. 'false', otherwise.
	 */
	public Boolean isJsonArray()
	{
		return new Boolean(this.value instanceof JsonArray);
	} // End of status method.
	
	/**
	 * Whether the value of this instance is a JSON object.
	 * 
	 * @return Boolean - 'true', if the value of this instance is a JSON object. 'false', otherwise.
	 */
	public Boolean isJsonObject()
	{
		return new Boolean(this.value instanceof JsonObject);
	} // End of status method.

	//*********************************************************************************************
	//
	//                  Overridden Methods of Class 'Object'
	//
	//*********************************************************************************************
	
	/**
	 * Whether this and another object are equal.
	 * 
	 * Note:
	 * This and another object are equal, if they are not 'null', instances of the same class (JsonValue) and are containing equal values.
	 * 
	 * @param Object obj - The other object.
	 * 
	 * @return boolean - 'true', if this and the other object are equal. 'false', otherwise.
	 */
	public boolean equals(Object obj)
	{
	    // Check, whether the other object is 'null'.
	    if(obj == null)
	    {
	        return false;
	    }
	    
	    // Check, whether the other object is an instance of the class JsonValue.
	    if(!(obj instanceof JsonValue))
	    {
	        return false;
	    }
	    
	    JsonValue other = (JsonValue)obj;
	    
	    // Return, whether the other object contains a value, which is equal to the value of this instance.
	    return ((this.isNull() && other.isNull()) || this.value.equals(other.getValue()));
	} // End of method 'equals'.
	
	/**
	 * Returns a JSON string of this JSON value.
	 */
	public String toString()
	{
		// Check, whether the value is 'null'.
		if(this.value == null)
		{
			return "null";
		}
		
		// Check, whether the value is a string.
		if(this.value instanceof String)
		{
			return "\"" + this.value.toString().replaceAll("\\\\{0}\\\"",
			                                               "\\\\\\\"") + "\"";
		}

		// For any other type, just call the toString() method.
		return this.value.toString();
	} // End of method 'toString'.

	//*********************************************************************************************
	//
	//                  Overridden Methods of Interface 'Cloneable'
	//
	//*********************************************************************************************
	
	/**
	 * Creates and returns a copy of this JSON value.
	 */
	public JsonValue clone()
	{
		return new JsonValue(this);
	} // End of method 'clone'.

} // End of class 'JsonValue'.
