
//*************************************************************************************************
//
//                      Package 'PDFrePRO'
//
//*************************************************************************************************

package PDFrePRO;

//*************************************************************************************************
//
//                      Imports
//
//*************************************************************************************************

// Package 'java.io'.
import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.InputStreamReader;

// Package 'java.net'.
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

// Package 'java.security'.
import java.security.MessageDigest;

// Package 'java.time'.
import java.time.ZonedDateTime;
import java.time.ZoneId;

// Package 'java.time.format'.
import java.time.format.DateTimeFormatter;

// Package 'java.util'.
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;

// Package 'java.util.concurrent.atomic'.
import java.util.concurrent.atomic.AtomicInteger;

// Package 'javax.crypto'.
import javax.crypto.Mac;

// Package 'javax.crypto.spec'.
import javax.crypto.spec.SecretKeySpec;

// Package 'json'.
import json.*;

//*************************************************************************************************
//
//                      Class 'PDFrePRO'
//
//*************************************************************************************************

/**
 * @description JAVA class to use the PDFrePRO print service via Internet (requires Java 8+).
 * @package 	PDFrePRO
 * @author  	RICHTER & POWELEIT GmbH
 * @see     	https://www.pdfrepro.de/
 * @version 	v3.03
 */
public class PDFrePRO
{

	//*********************************************************************************************
	//
	//                  Addresses
	//
	//*********************************************************************************************

    /**
     * The PDFrePRO host, where all requests will be send to.
     */
	private String host = "https://api.pdfrepro.de";

    //*********************************************************************************************
    //
    //                  Credentials
    //
    //*********************************************************************************************

    /**
     * The API key, which shall be used for requests.
     */
    private String apiKey    = "";

    /**
     * The associated shared key to 'apiKey'.
     */
    private String sharedKey = "";

	//*********************************************************************************************
	//
	//                  URIs
	//
	//*********************************************************************************************

    /**
     * The URI for general requests on placeholders.
     */
    private static final String URI_PLACEHOLDERS              = "/v3/placeholders";

    /**
     * The URI for requests on a specific placeholder.
     */
    private static final String URI_PLACEHOLDERS_ID           = "/v3/placeholders/{id}";

    /**
     * The URI for requests on templates, which are using a specific placeholder.
     */
    private static final String URI_PLACEHOLDERS_ID_TEMPLATES = "/v3/placeholders/{id}/templates";

    /**
     * The URI for general requests on templates.
     */
    private static final String URI_TEMPLATES                 = "/v3/templates";

    /**
     * The URI for requests on a specific template.
     */
    private static final String URI_TEMPLATES_ID              = "/v3/templates/{id}";

    /**
     * The URI for requests on placeholders, which are used by a specific template.
     */
    private static final String URI_TEMPLATES_ID_PLACEHOLDERS = "/v3/templates/{id}/placeholders";

    /**
     * The URI for requests on the WYSIWYG-editor, which is using a specific template.
     */
    private static final String URI_TEMPLATES_ID_EDITOR_URL   = "/v3/templates/{id}/editor-url";

    /**
     * The URI for requests on the PDF of a specific template.
     */
    private static final String URI_TEMPLATES_ID_PDF          = "/v3/templates/{id}/pdf";

    //*********************************************************************************************
    //
    //                  Validations
    //
    //*********************************************************************************************

    /**
     * This array contains all valid response statuses.
     */
    private static final String[] VALID_STATUSES     = {
						                                   "success",
						                                   "error",
						                                   "fail"
    												   };

    /**
     * This array contains all valid status codes for errors and fails.
     */
    private static final int[] 	  VALID_STATUS_CODES = {
    												       400,
    												       401,
    												       404,
    												       405,
    												       406,
    												       408,
    												       409,
    												       411,
    												       500
    												   };

    //*********************************************************************************************
    //
    //                  Constructors
    //
    //*********************************************************************************************

    /**
     * PDFrePRO constructor.
     *
     * @param String apiKey    - The API key, which shall be used for requests.
     * @param String sharedKey - The associated shared key to 'apiKey'.
     *
     * @throws PDFrePROException - If 'apiKey' or 'sharedKey' are invalid.
     */
    public PDFrePRO(String apiKey, String sharedKey) throws PDFrePROException
    {
    	this(apiKey, sharedKey, "");
    }

    /**
     * PDFrePRO constructor.
     *
     * @param String apiKey    - The API key, which shall be used for requests.
     * @param String sharedKey - The associated shared key to 'apiKey'.
     * @param string host      - An optional API host, which shall be used for requests.
     *
     * @throws PDFrePROException - If 'apiKey' or 'sharedKey' are invalid.
     */
    public PDFrePRO(String apiKey, String sharedKey, String host) throws PDFrePROException
    {
        // Set keys.
        this.setApiKey(apiKey);
        this.setSharedKey(sharedKey);

        // Set host.
        if (!host.isEmpty()) {
            this.setHost(host);
        }
    } // End of constructor.

    //*********************************************************************************************
    //
    //                  Setter
    //
    //*********************************************************************************************

    /**
     * Sets the API key, which shall be used for requests.
     *
     * @param String apiKey - The API key, which shall be used for requests.
     *
     * @throws PDFrePROException - If 'apiKey' is invalid.
     */
    public void setApiKey(String apiKey) throws PDFrePROException
    {
        if((apiKey.length() != 20) || !apiKey.matches("[a-zA-Z0-9]{20}"))
        {
            throw new PDFrePROException("Invalid apiKey!");
        }

        this.apiKey = apiKey;
    } // End of setter.

    /**
     * Sets the PDFrePRO host, which shall be used for requests.
     *
     * @param string $host - The PDFrePRO host, which shall be used for requests.
     *
     * @throws PDFrePROException - If 'host' is invalid.
     */
    public void setHost(String host) throws PDFrePROException
    {
        if (host.isEmpty()) {
            throw new PDFrePROException("Invalid host!");
        }

        this.host = host;
    }

    /**
     * Sets the associated shared key to 'apiKey'.
     *
     * @param String sharedKey - The associated shared key to 'apiKey'.
     *
     * @throws PDFrePROException - If 'sharedKey' is invalid.
     */
    public void setSharedKey(String sharedKey) throws PDFrePROException
    {
        if((sharedKey.length() != 64) || !sharedKey.matches("[a-zA-Z0-9]{64}"))
        {
            throw new PDFrePROException("Invalid sharedKey!");
        }

        this.sharedKey = sharedKey;
    } // End of setter.

    //*********************************************************************************************
    //
    //                  Functions for Placeholders
    //
    //*********************************************************************************************

    /**
     * Creates a new placeholder for your PDFrePRO account.
     *
     * @param String name - The name of the new placeholder.
     * @param String data - The data of the new placeholder.
     *
     * @return JsonValue - A relative URL to the new placeholder; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue createPlaceholder(String name,
                                       String data) throws PDFrePROException
    {
        return this.createPlaceholder(name,
                                      data,
                                      false);
    } // End of method 'createPlaceholder'.

    /**
     * Creates a new placeholder for your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  name                 - The name of the new placeholder.
     * @param String  data                 - The data of the new placeholder.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the relative URL to
     *                                           the new placeholder.
     *
     * @return JsonValue - A relative URL to the new placeholder; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue createPlaceholder(String  name,
                                       String  data,
                                       boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        JsonObject content = new JsonObject();

        content.put("name",
                    new JsonValue(name));
        content.put("data",
                    new JsonValue(data));

        JsonValue response = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS,
                                              "POST",
                                              new JsonValue(content));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      201
                                  });

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }

        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");

        if(!url.isString() || !url.getStringValue().substring(0,
                                                              PDFrePRO.URI_PLACEHOLDERS.length() + 1).equals(PDFrePRO.URI_PLACEHOLDERS + "/"))
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return url;
    } // End of method 'createPlaceholder'.

    /**
     * Gets a specific placeholder of your PDFrePRO account.
     *
     * @param String id - The ID of the placeholder, which shall be requested.
     *
     * @return JsonValue - The requested placeholder; or the entire response, if 'returnEntireResponse' is set to
     *                         'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPlaceholder(String id) throws PDFrePROException
    {
        return this.getPlaceholder(id,
                                   false);
    } // End of method 'getPlaceholder'.

    /**
     * Gets a specific placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the placeholder, which shall be requested.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the requested
     *                                           placeholder.
     *
     * @return JsonValue - The requested placeholder; or the entire response, if 'returnEntireResponse' is set to
     *                         'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPlaceholder(String  id,
                                    boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS_ID.replaceAll("\\{id\\}",
                                                                                      id));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response);

        // Check, whether the placeholder is valid.
        JsonObject data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!data.containsKey("id") || !data.get("id").isString()
                                   || !data.get("id").getStringValue().equals(id) ||
           !data.containsKey("name")                                              ||
           !data.containsKey("lastModificationDate")                              ||
           !data.containsKey("rawData")                                           ||
           !data.containsKey("numberOfReferencedTemplates") || !data.get("numberOfReferencedTemplates").isNumber()
                                                            || (data.get("numberOfReferencedTemplates").getNumberValue().intValue() < 0))
        {
            throw new PDFrePROException("Response is invalid, due to invalid placeholder!");
        }

        return new JsonValue(data);
    } // End of method 'getPlaceholder'.

    /**
     * Gets all templates of your PDFrePRO account, which are using a specific placeholder.
     *
     * @param String id - The ID of the placeholder, which is used by the requested templates.
     *
     * @return JsonValue - The requested templates; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getTemplatesByPlaceholder(String id) throws PDFrePROException
    {
        return this.getTemplatesByPlaceholder(id,
                                              false);
    } // End of method 'getTemplatesByPlaceholder'.

    /**
     * Gets all templates of your PDFrePRO account, which are using a specific placeholder.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the placeholder, which is used by the requested templates.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the requested
     *                                           templates.
     *
     * @return JsonValue - The requested templates; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getTemplatesByPlaceholder(String  id,
                                               boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        AtomicInteger responseCode = new AtomicInteger();
        JsonValue     response     = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS_ID_TEMPLATES.replaceAll("\\{id\\}",
                                                                                                        id),
                                                      "GET",
                                                      new JsonValue(new JsonObject()),
                                                      responseCode);

        // Check, whether the HTTP status code indicates no content.
        if(responseCode.intValue() == 204)
        {
            // Build a valid response body.
            JsonObject temp = new JsonObject();

            temp.put("code",
                     new JsonValue(204));
            temp.put("status",
                     new JsonValue("success"));
            temp.put("data",
                     new JsonValue(new JsonObject("templates",
                                                  new JsonValue(new JsonArray()))));
            response.setValue(temp);
        }

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      200,
                                      204
                                  });

        // Check, whether templates are available.
        JsonObject data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!data.containsKey("templates") || !data.get("templates").isJsonArray())
        {
            throw new PDFrePROException("Response is invalid, due to missing or invalid templates!");
        }

        // Check, whether all templates are valid.
        for(JsonValue template : data.get("templates").getJsonArrayValue())
        {
            // Check, whether the template is a JSON object.
            if(!template.isJsonObject())
            {
                throw new PDFrePROException("Response is invalid, due to invalid template!");
            }

            // Check, whether the template is valid.
            JsonObject temp = template.getJsonObjectValue();

            if (!temp.containsKey("id") || !temp.containsKey("name") || !temp.containsKey("lastModificationDate")) {
                throw new PDFrePROException("Response is invalid, due to invalid template!");
            }
        }

        return data.get("templates");
    } // End of method 'getTemplatesByPlaceholder'.

    /**
     * Gets all placeholders of your PDFrePRO account.
     *
     * @return JsonValue - All placeholders of your PDFrePRO account; or the entire response, if 'returnEntireResponse'
     *                         is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getAllPlaceholders() throws PDFrePROException
    {
        return this.getAllPlaceholders(false);
    } // End of method 'getAllPlaceholders'.

    /**
     * Gets all placeholders of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the found
     *                                           placeholders.
     *
     * @return JsonValue - All placeholders of your PDFrePRO account; or the entire response, if 'returnEntireResponse'
     *                         is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getAllPlaceholders(boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        AtomicInteger responseCode = new AtomicInteger();
        JsonValue     response     = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS,
                                                      "GET",
                                                      new JsonValue(new JsonObject()),
                                                      responseCode);

        // Check, whether the HTTP status code indicates no content.
        if(responseCode.intValue() == 204)
        {
            // Build a valid response body.
            JsonObject temp = new JsonObject();

            temp.put("code",
                     new JsonValue(204));
            temp.put("status",
                     new JsonValue("success"));
            temp.put("data",
                     new JsonValue(new JsonObject("placeholders",
                                                  new JsonValue(new JsonArray()))));
            response.setValue(temp);
        }

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      200,
                                      204
                                  });

        // Check, whether placeholders are available.
        JsonObject data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!data.containsKey("placeholders") || !data.get("placeholders").isJsonArray())
        {
            throw new PDFrePROException("Response is invalid, due to missing or invalid placeholders!");
        }

        // Check, whether all placeholders are valid.
        for(JsonValue placeholder : data.get("placeholders").getJsonArrayValue())
        {
            // Check, whether the placeholder is a JSON object.
            if(!placeholder.isJsonObject())
            {
                throw new PDFrePROException("Response is invalid, due to invalid placeholder!");
            }

            // Check, whether the placeholder is valid.
            JsonObject temp = placeholder.getJsonObjectValue();

            if(!temp.containsKey("id")                   ||
               !temp.containsKey("name")                 ||
               !temp.containsKey("lastModificationDate") ||
               !temp.containsKey("numberOfReferencedTemplates") || !temp.get("numberOfReferencedTemplates").isNumber()
                                                                || (temp.get("numberOfReferencedTemplates").getNumberValue().intValue() < 0))
            {
                throw new PDFrePROException("Response is invalid, due to invalid placeholder!");
            }
        }

        return data.get("placeholders");
    } // End of method 'getAllPlaceholders'.

    /**
     * Updates an existing placeholder of your PDFrePRO account.
     *
     * @param String id   - The ID of the placeholder, which shall be updated.
     * @param String name - The new name of the placeholder.
     *
     * @return JsonValue - Whether the placeholder got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updatePlaceholder(String id,
                                       String name) throws PDFrePROException
    {
        return this.updatePlaceholder(id,
                                      name,
                                      "",
                                      false);
    } // End of method 'updatePlaceholder'.

    /**
     * Updates an existing placeholder of your PDFrePRO account.
     *
     * @param String id   - The ID of the placeholder, which shall be updated.
     * @param String name - The new name of the placeholder.
     * @param String data - The new data of the placeholder.
     *
     * @return JsonValue - Whether the placeholder got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updatePlaceholder(String id,
                                       String name,
                                       String data) throws PDFrePROException
    {
        return this.updatePlaceholder(id,
                                      name,
                                      data,
                                      false);
    } // End of method 'updatePlaceholder'.

    /**
     * Updates an existing placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the placeholder, which shall be updated.
     * @param String  name                 - The new name of the placeholder.
     * @param String  data                 - The new data of the placeholder.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just whether the
     *                                           placeholder got updated.
     *
     * @return JsonValue - Whether the placeholder got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updatePlaceholder(String  id,
                                       String  name,
                                       String  data,
                                       boolean returnEntireResponse) throws PDFrePROException
    {
        // Prepare the request.
        JsonObject content = new JsonObject();

        if((name != null) && !name.isEmpty())
        {
            content.put("name",
                        new JsonValue(name));
        }
        if((data != null) && !data.isEmpty())
        {
            content.put("data",
                        new JsonValue(data));
        }

        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS_ID.replaceAll("\\{id\\}",
                                                                                      id),
                                              "PUT",
                                              new JsonValue(content));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response);

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }

        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");

        if(!url.isString() || !url.getStringValue().equals(PDFrePRO.URI_PLACEHOLDERS_ID.replaceAll("\\{id\\}",
                                                                                                   id)))
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return new JsonValue(response.getJsonObjectValue().get("status").getStringValue().equals("success"));
    } // End of method 'updatePlaceholder'.

    /**
     * Copies an existing placeholder of your PDFrePRO account.
     *
     * @param String id - The ID of the placeholder, which shall be copied.
     *
     * @return JsonValue - A relative URL to the copied placeholder; or the entire response, if 'returnEntireResponse'
     *                         is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyPlaceholder(String id) throws PDFrePROException
    {
        return this.copyPlaceholder(id,
                                    "",
                                    false);
    } // End of method 'copyPlaceholder'.

    /**
     * Copies an existing placeholder of your PDFrePRO account.
     *
     * @param String id   - The ID of the placeholder, which shall be copied.
     * @param String name - The name of the copied placeholder.
     *
     * @return JsonValue - A relative URL to the copied placeholder; or the entire response, if 'returnEntireResponse'
     *                         is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyPlaceholder(String id,
                                     String name) throws PDFrePROException
    {
        return this.copyPlaceholder(id,
                                    name,
                                    false);
    } // End of method 'copyPlaceholder'.

    /**
     * Copies an existing placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the placeholder, which shall be copied.
     * @param String  name                 - The name of the copied placeholder.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the relative URL to
     *                                           the copied placeholder.
     *
     * @return JsonValue - A relative URL to the copied placeholder; or the entire response, if 'returnEntireResponse'
     *                         is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyPlaceholder(String  id,
                                     String  name,
                                     boolean returnEntireResponse) throws PDFrePROException
    {
        // Prepare the request.
        JsonObject content = new JsonObject();

        if((name != null) && !name.isEmpty())
        {
            content.put("name",
                        new JsonValue(name));
        }

        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS_ID.replaceAll("\\{id\\}",
                                                                                      id),
                                              "POST",
                                              new JsonValue(content));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      201
                                  });

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }

        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");

        if(!url.isString() || !url.getStringValue().substring(0,
                                                              PDFrePRO.URI_PLACEHOLDERS.length() + 1).equals(PDFrePRO.URI_PLACEHOLDERS + "/"))
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return url;
    } // End of method 'copyPlaceholder'.

    /**
     * Deletes an existing placeholder of your PDFrePRO account.
     *
     * @param String id - The ID of the placeholder, which shall be deleted.
     *
     * @return JsonValue - Whether the placeholder got deleted ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue deletePlaceholder(String id) throws PDFrePROException
    {
        return this.deletePlaceholder(id,
                                      false);
    } // End of method 'deletePlaceholder'.

    /**
     * Deletes an existing placeholder of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the placeholder, which shall be deleted.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just whether the
     *                                           placeholder got deleted.
     *
     * @return JsonValue - Whether the placeholder got deleted ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue deletePlaceholder(String  id,
                                       boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        AtomicInteger responseCode = new AtomicInteger();
        JsonValue     response     = this.sendRequest(PDFrePRO.URI_PLACEHOLDERS_ID.replaceAll("\\{id\\}",
                                                                                              id),
                                                      "DELETE",
                                                      new JsonValue(new JsonObject()),
                                                      responseCode);

        // Check, whether the HTTP status code indicates no content.
        if(responseCode.intValue() == 204)
        {
            // Build a valid response body.
            JsonObject temp = new JsonObject();

            temp.put("code",
                     new JsonValue(204));
            temp.put("status",
                     new JsonValue("success"));
            temp.put("data",
                     new JsonValue(new JsonObject()));
            response.setValue(temp);
        }

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      204
                                  });

        // Return the result.
        return new JsonValue(response.getJsonObjectValue().get("status").getStringValue().equals("success"));
    } // End of method 'deletePlaceholder'.

    //*********************************************************************************************
    //
    //                  Functions for Templates
    //
    //*********************************************************************************************

    /**
     * Creates a new template for your PDFrePRO account.
     *
     * @param String name - The name of the new template.
     *
     * @return JsonValue - A relative URL to the new template; or the entire response, if 'returnEntireResponse' is set
     *                         to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue createTemplate(String name) throws PDFrePROException
    {
        return this.createTemplate(name,
                                   "",
                                   new String[0],
                                   false);
    } // End of method 'createTemplate'.

    /**
     * Creates a new template for your PDFrePRO account.
     *
     * @param String name        - The name of the new template.
     * @param String description - The description of the new template.
     *
     * @return JsonValue - A relative URL to the new template; or the entire response, if 'returnEntireResponse' is set
     *                         to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue createTemplate(String name,
                                    String description) throws PDFrePROException
    {
        return this.createTemplate(name,
                                   description,
                                   new String[0],
                                   false);
    } // End of method 'createTemplate'.

    /**
     * Creates a new template for your PDFrePRO account.
     *
     * @param String   name           - The name of the new template.
     * @param String   description    - The description of the new template.
     * @param String[] placeholderIds - The IDs of all placeholders, which shall be used by the new template.
     *
     * @return JsonValue - A relative URL to the new template; or the entire response, if 'returnEntireResponse' is set
     *                         to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue createTemplate(String   name,
                                    String   description,
                                    String[] placeholderIds) throws PDFrePROException
    {
        return this.createTemplate(name,
                                   description,
                                   placeholderIds,
                                   false);
    } // End of method 'createTemplate'.

    /**
     * Creates a new template for your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String   name                 - The name of the new template.
     * @param String   description          - The description of the new template.
     * @param String[] placeholderIds       - The IDs of all placeholders, which shall be used by the new template.
     * @param boolean  returnEntireResponse - Whether the entire response shall be returned, or just the relative URL to
     *                                            the new template.
     *
     * @return JsonValue - A relative URL to the new template; or the entire response, if 'returnEntireResponse' is set
     *                         to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue createTemplate(String   name,
                                    String   description,
                                    String[] placeholderIds,
                                    boolean  returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        JsonObject content = new JsonObject();
        JsonArray  ids     = new JsonArray();

        for(String id : placeholderIds)
        {
            ids.add(new JsonValue(id));
        }

        content.put("name", new JsonValue(name));
        content.put("description", new JsonValue(description));
        content.put("placeholderIds", new JsonValue(ids));

        JsonValue response = this.sendRequest(PDFrePRO.URI_TEMPLATES,
                                              "POST",
                                              new JsonValue(content));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      201
                                  });

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }

        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");

        if(!url.isString() || !url.getStringValue().substring(0,
                                                              PDFrePRO.URI_TEMPLATES.length() + 1).equals(PDFrePRO.URI_TEMPLATES + "/"))
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return url;
    } // End of method 'createTemplate'.

    /**
     * Gets a specific template of your PDFrePRO account.
     *
     * @param String id - The ID of the template, which shall be requested.
     *
     * @return JsonValue - The requested template; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getTemplate(String id) throws PDFrePROException
    {
        return this.getTemplate(id,
                                false);
    } // End of method 'getTemplate'.

    /**
     * Gets a specific template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the template, which shall be requested.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the requested
     *                                           template.
     *
     * @return JsonValue - The requested template; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getTemplate(String  id,
                                 boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID.replaceAll("\\{id\\}",
                                                                                   id));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response);

        // Check, whether the template is valid.
        JsonObject data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if (
            !data.containsKey("id")   || !data.get("id").isString() || !data.get("id").getStringValue().equals(id) ||
            !data.containsKey("name") || !data.containsKey("lastModificationDate")
        ) {
            throw new PDFrePROException("Response is invalid, due to invalid template!");
        }

        return new JsonValue(data);
    } // End of method 'getTemplate'.

    /**
     * Gets all placeholders of your PDFrePRO account, which are used by a specific template.
     *
     * @param String id - The ID of the template, which uses the requested placeholders.
     *
     * @return JsonValue - The requested placeholders; or the entire response, if 'returnEntireResponse' is set to
     *                         'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPlaceholdersByTemplate(String id) throws PDFrePROException
    {
        return this.getPlaceholderIDsByTemplate(id,
                                                false);
    } // End of method 'getPlaceholderIDsByTemplate'.

    /**
     * Gets all placeholders of your PDFrePRO account, which are used by a specific template.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the template, which uses the requested placeholders.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the requested
     *                                           placeholders.
     *
     * @return JsonValue - The requested placeholders; or the entire response, if 'returnEntireResponse' is set to
     *                         'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPlaceholderIDsByTemplate(String  id,
                                                 boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        AtomicInteger responseCode = new AtomicInteger();
        JsonValue     response     = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID_PLACEHOLDERS.replaceAll("\\{id\\}",
                                                                                                        id),
                                                      "GET",
                                                      new JsonValue(new JsonObject()),
                                                      responseCode);

        // Check, whether the HTTP status code indicates no content.
        if(responseCode.intValue() == 204)
        {
            // Build a valid response body.
            JsonObject temp = new JsonObject();

            temp.put("code",
                     new JsonValue(204));
            temp.put("status",
                     new JsonValue("success"));
            temp.put("data",
                     new JsonValue(new JsonObject("placeholders",
                                                  new JsonArray())));
            response.setValue(temp);
        }

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      200,
                                      204
                                  });

        // Check, whether placeholders are available.
        JsonObject data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!data.containsKey("placeholders") || !data.get("placeholders").isJsonArray())
        {
            throw new PDFrePROException("Response is invalid, due to missing or invalid placeholder IDs!");
        }

        // Check, whether all placeholders are valid.
        JsonArray placeholders = data.get("placeholders").getJsonArrayValue();

        for(JsonValue placeholder : placeholders)
        {
            // Check, whether the placeholder is a JSON object.
            if(!placeholder.isJsonObject())
            {
                throw new PDFrePROException("Response is invalid, due to invalid placeholder!");
            }

            // Check, whether the placeholder is valid.
            JsonObject temp = placeholder.getJsonObjectValue();

            if(!temp.containsKey("id")                   ||
               !temp.containsKey("name")                 ||
               !temp.containsKey("lastModificationDate") ||
               !temp.containsKey("numberOfReferencedTemplates") || !temp.get("numberOfReferencedTemplates").isNumber()
                                                                || (temp.get("numberOfReferencedTemplates").getNumberValue().intValue() < 0))
            {
                throw new PDFrePROException("Response is invalid, due to invalid placeholder!");
            }
        }

        return new JsonValue(placeholders);
    } // End of method 'getPlaceholdersByTemplate'.

    /**
     * Gets all templates of your PDFrePRO account.
     *
     * @return JsonValue - All templates of your PDFrePRO account; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getAllTemplates() throws PDFrePROException
    {
        return this.getAllTemplates(false);
    } // End of method 'getAllTemplates'.

    /**
     * Gets all templates of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the found templates.
     *
     * @return JsonValue - All templates of your PDFrePRO account; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getAllTemplates(boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        AtomicInteger responseCode = new AtomicInteger();
        JsonValue     response     = this.sendRequest(PDFrePRO.URI_TEMPLATES,
                                                      "GET",
                                                      new JsonValue(new JsonObject()),
                                                      responseCode);

        // Check, whether the HTTP status code indicates no content.
        if(responseCode.intValue() == 204)
        {
            // Build a valid response body.
            JsonObject temp = new JsonObject();

            temp.put("code",
                     new JsonValue(204));
            temp.put("status",
                     new JsonValue("success"));
            temp.put("data",
                     new JsonValue(new JsonObject("templates",
                                                  new JsonArray())));
            response.setValue(temp);
        }

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      200,
                                      204
                                  });

        // Check, whether templates are available.
        JsonObject data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!data.containsKey("templates") || !data.get("templates").isJsonArray())
        {
            throw new PDFrePROException("Response is invalid, due to missing or invalid templates!");
        }

        // Check, whether all templates are valid.
        JsonArray templates = data.get("templates").getJsonArrayValue();

        for(JsonValue element : templates)
        {
            if(!element.isJsonObject())
            {
                throw new PDFrePROException("Response is invalid, due to invalid template!");
            }

            JsonObject template = element.getJsonObjectValue();

            if (!template.containsKey("id") || !template.containsKey("name") || !template.containsKey("lastModificationDate")) {
                throw new PDFrePROException("Response is invalid, due to invalid template!");
            }
        }

        return new JsonValue(templates);
    } // End of method 'getAllTemplates'.

    /**
     * Gets an URL, which opens the WYSIWYG editor, to edit an existing template of your PDFrePRO account.
     *
     * IMPORTANT NOTE:
     * Before you start using our PDFrePRO-API, please make sure you have everything
     * configured correctly in your account in our Portal (https://portal.pdfrepro.de).
     * When you are intending to integrate our PDFrePRO-WYSIWYG-Editor into your own
     * application you need to set up a success- and abort-url as follow-up urls to
     * let the Editor know where to redirect to when you want to leave the editor.
     *
     * @param String id - The ID of the template, which shall be opened in the WYSIWYG editor.
     *
     * @return JsonValue - An URL to open the WYSIWYG editor; or the entire response, if 'returnEntireResponse' is set
     *                         to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getEditorUrl(String id) throws PDFrePROException
    {
        return this.getEditorUrl(id,
                                 false);
    } // End of method 'getEditorUrl'.

    /**
     * Gets an URL, which opens the WYSIWYG editor, to edit an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * IMPORTANT NOTE:
     * Before you start using our PDFrePRO-API, please make sure you have everything
     * configured correctly in your account in our Portal (https://portal.pdfrepro.de).
     * When you are intending to integrate our PDFrePRO-WYSIWYG-Editor into your own
     * application you need to set up a success- and abort-url as follow-up urls to
     * let the Editor know where to redirect to when you want to leave the editor.
     *
     * @param String  id                   - The ID of the template, which shall be opened in the WYSIWYG editor.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the URL to the
     *                                           WYSIWYG editor.
     *
     * @return JsonValue - An URL to open the WYSIWYG editor; or the entire response, if 'returnEntireResponse' is set
     *                         to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getEditorUrl(String  id,
                                  boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID_EDITOR_URL.replaceAll("\\{id\\}",
                                                                                              id));

        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response);

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();

        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }

        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");

        if(!url.isString())
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return url;
    } // End of method 'getEditorUrl'.

    /**
     * Gets a Base64-encoded PDF of an existing template of your PDFrePRO account.
     *
     * @param String id - The ID of the template, which shall be printed as PDF.
     *
     * @return JsonValue - A Base64-encoded PDF; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPDF(String id) throws PDFrePROException
    {
        return this.getPDF(id,
                           new JsonValue(new JsonObject()),
                           "",
                           false);
    } // End of method 'getPDF'.

    /**
     * Gets a Base64-encoded PDF of an existing template of your PDFrePRO account.
     *
     * @param String    id                   - The ID of the template, which shall be printed as PDF.
     * @param JsonValue data                 - The data for the placeholders, which are used by the template.
     *                                         Example: new JsonValue(new JsonObject(<placeholderName>,
     *                                                                               new JsonValue(new JsonObject(<placeholderDataName>,
     *                                                                                                            new JsonValue(<placeholderDataValue>)))));
     *
     * @return JsonValue - A Base64-encoded PDF; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPDF(String    id,
                            JsonValue data) throws PDFrePROException
    {
        return this.getPDF(id,
                           data,
                           "",
                           false);
    } // End of method 'getPDF'.

    /**
     * Gets a Base64-encoded PDF of an existing template of your PDFrePRO account.
     *
     * @param String    id                   - The ID of the template, which shall be printed as PDF.
     * @param JsonValue data                 - The data for the placeholders, which are used by the template.
     *                                         Example: new JsonValue(new JsonObject(<placeholderName>,
     *                                                                               new JsonValue(new JsonObject(<placeholderDataName>,
     *                                                                                                            new JsonValue(<placeholderDataValue>)))));
     * @param String    language             - Any language, you have defined in settings of your API key.
     *                                         Example: 'en'
     *
     * @return JsonValue - A Base64-encoded PDF; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPDF(String    id,
                            JsonValue data,
                            String    language) throws PDFrePROException
    {
        return this.getPDF(id,
                           data,
                           language,
                           false);
    } // End of method 'getPDF'.

    /**
     * Gets a Base64-encoded PDF of an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String    id                   - The ID of the template, which shall be printed as PDF.
     * @param JsonValue data                 - The data for the placeholders, which are used by the template.
     *                                         Example: new JsonValue(new JsonObject(<placeholderName>,
     *                                                                               new JsonValue(new JsonObject(<placeholderDataName>,
     *                                                                                                            new JsonValue(<placeholderDataValue>)))));
     * @param String    language             - Any language, you have defined in settings of your API key.
     *                                         Example: 'en'
     * @param boolean   returnEntireResponse - Whether the entire response shall be returned, or just the PDF.
     *
     * @return JsonValue - A Base64-encoded PDF; or the entire response, if 'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue getPDF(String    id,
                            JsonValue data,
                            String    language,
                            boolean   returnEntireResponse) throws PDFrePROException
    {
        // Prepare the request.
        if(data == null)
        {
            data = new JsonValue(new JsonObject());
        }

        Map<String, JsonValue> map = new TreeMap<String, JsonValue>();

        map.put("data",     new JsonValue(data.toString()));
        map.put("language", new JsonValue(language));

        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID_PDF.replaceAll("\\{id\\}",
                                                                                       id),
                                              "POST",
                                              new JsonValue(new JsonObject(map)));
        
        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      201, 429
                                  });
        
        // Check, whether the PDF is available and valid.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();
        
        if(!_data.containsKey("pdf") || !_data.get("pdf").isString())
        {
            throw new PDFrePROException("Response is invalid, due to invalid PDF!");
        }
        
        return _data.get("pdf");
    } // End of method 'getPDF'.

    /**
     * Updates an existing template of your PDFrePRO account.
     *
     * @param String id   - The ID of the template, which shall be updated.
     * @param String name - The new name of the template.
     *
     * @return JsonValue - Whether the template got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updateTemplate(String id,
                                    String name) throws PDFrePROException
    {
        return this.updateTemplate(id,
                                   name,
                                   "",
                                   null,
                                   false);
    } // End of method 'updateTemplate'.

    /**
     * Updates an existing template of your PDFrePRO account.
     *
     * @param String id          - The ID of the template, which shall be updated.
     * @param String name        - The new name of the template.
     * @param String description - The new description of the template.
     *
     * @return JsonValue - Whether the template got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updateTemplate(String id,
                                    String name,
                                    String description) throws PDFrePROException
    {
        return this.updateTemplate(id,
                                   name,
                                   description,
                                   null,
                                   false);
    } // End of method 'updateTemplate'.

    /**
     * Updates an existing template of your PDFrePRO account.
     *
     * @param String   id             - The ID of the template, which shall be updated.
     * @param String   name           - The new name of the template.
     * @param String   description    - The new description of the template.
     * @param String[] placeholderIds - The IDs of all placeholders, which shall be used by the template. (Removes all
     *                                      existing usages of placeholders by the template.)
     *
     * @return JsonValue - Whether the template got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updateTemplate(String   id,
                                    String   name,
                                    String   description,
                                    String[] placeholderIds) throws PDFrePROException
    {
        return this.updateTemplate(id,
                                   name,
                                   description,
                                   placeholderIds,
                                   false);
    } // End of method 'updateTemplate'.

    /**
     * Updates an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String   id                   - The ID of the template, which shall be updated.
     * @param String   name                 - The new name of the template.
     * @param String   description          - The new description of the template.
     * @param String[] placeholderIds       - The IDs of all placeholders, which shall be used by the template. (Removes
     *                                            all existing usages of placeholders by the template.)
     * @param boolean  returnEntireResponse - Whether the entire response shall be returned, or just whether the
     *                                            template got updated.
     *
     * @return JsonValue - Whether the template got updated ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue updateTemplate(String   id,
                                    String   name,
                                    String   description,
                                    String[] placeholderIds,
                                    boolean  returnEntireResponse) throws PDFrePROException
    {
        // Prepare the request.
        JsonObject content = new JsonObject();
        
        if((name != null) && !name.isEmpty())
        {
            content.put("name",
                        new JsonValue(name));
        }
        if((description != null) && !description.isEmpty())
        {
            content.put("description",
                        new JsonValue(description));
        }
        if(placeholderIds != null)
        {
            JsonArray ids = new JsonArray();
            
            for(String placeholderId : placeholderIds)
            {
                ids.add(new JsonValue(placeholderId));
            }
            
            content.put("placeholderIds",
                        new JsonValue(ids));
        }
        
        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID.replaceAll("\\{id\\}",
                                                                                   id),
                                              "PUT",
                                              new JsonValue(content));
        
        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response);

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();
        
        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }
        
        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");
        
        if(!url.isString() || !url.getStringValue().equals(PDFrePRO.URI_TEMPLATES_ID.replaceAll("\\{id\\}", 
                                                                                                id)))
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return new JsonValue(response.getJsonObjectValue().get("status").getStringValue().equals("success"));
    } // End of method 'updateTemplate'.

    /**
     * Copies an existing template of your PDFrePRO account.
     *
     * @param String id - The ID of the template, which shall be copied.
     *
     * @return JsonValue - A relative URL to the copied template; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyTemplate(String id) throws PDFrePROException
    {
        return this.copyTemplate(id,
                                 "",
                                 "",
                                 false);
    } // End of method 'copyTemplate'.

    /**
     * Copies an existing template of your PDFrePRO account.
     *
     * @param String id   - The ID of the template, which shall be copied.
     * @param String name - The name of the copied template.
     *
     * @return JsonValue - A relative URL to the copied template; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyTemplate(String id,
                                  String name) throws PDFrePROException
    {
        return this.copyTemplate(id,
                                 name,
                                 "",
                                 false);
    } // End of method 'copyTemplate'.

    /**
     * Copies an existing template of your PDFrePRO account.
     *
     * @param String id          - The ID of the template, which shall be copied.
     * @param String name        - The name of the copied template.
     * @param String description - The description of the copied template.
     *
     * @return JsonValue - A relative URL to the copied template; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyTemplate(String id,
                                  String name,
                                  String description) throws PDFrePROException
    {
        return this.copyTemplate(id,
                                 name,
                                 description,
                                 false);
    } // End of method 'copyTemplate'.

    /**
     * Copies an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the template, which shall be copied.
     * @param String  name                 - The name of the copied template.
     * @param String  description          - The description of the copied template.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just the relative URL to
     *                                           the copied template.
     *
     * @return JsonValue - A relative URL to the copied template; or the entire response, if 'returnEntireResponse' is
     *                         set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue copyTemplate(String  id,
                                  String  name,
                                  String  description,
                                  boolean returnEntireResponse) throws PDFrePROException
    {
        // Prepare the request.
        JsonObject content = new JsonObject();
        
        if((name != null) && !name.isEmpty())
        {
            content.put("name",
                        new JsonValue(name));
        }
        if((description != null) && !description.isEmpty())
        {
            content.put("description",
                        new JsonValue(description));
        }
        
        // Send the request.
        JsonValue response = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID.replaceAll("\\{id\\}",
                                                                                   id),
                                              "POST",
                                              new JsonValue(content));
        
        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      201
                                  });

        // Check, whether the relative URL is available.
        JsonObject _data = response.getJsonObjectValue().get("data").getJsonObjectValue();
        
        if(!_data.containsKey("url"))
        {
            throw new PDFrePROException("Response is invalid, due to missing URL!");
        }
        
        // Check, whether the relative URL is valid.
        JsonValue url = _data.get("url");
        
        if(!url.isString() || !url.getStringValue().substring(0,
                                                              PDFrePRO.URI_TEMPLATES.length() + 1).equals(PDFrePRO.URI_TEMPLATES + "/"))
        {
            throw new PDFrePROException("Response is invalid, due to invalid URL!");
        }

        return url;
    } // End of method 'copyTemplate'.

    /**
     * Deletes an existing template of your PDFrePRO account.
     *
     * @param String id - The ID of the template, which shall be deleted.
     *
     * @return JsonValue - Whether the template got deleted ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue deleteTemplate(String id) throws PDFrePROException
    {
        return this.deleteTemplate(id,
                                   false);
    } // End of method 'deleteTemplate'.

    /**
     * Deletes an existing template of your PDFrePRO account.
     *
     * Note:
     * If the entire response shall be returned, it will be returned directly after it got received (without
     * validation).
     *
     * @param String  id                   - The ID of the template, which shall be deleted.
     * @param boolean returnEntireResponse - Whether the entire response shall be returned, or just whether the template
     *                                           got deleted.
     *
     * @return JsonValue - Whether the template got deleted ('true', 'false'); or the entire response, if
     *                         'returnEntireResponse' is set to 'true'.
     *
     * @throws PDFrePROException - If the request could not be send, properly, or the response is invalid or contains an
     *                                 error.
     */
    public JsonValue deleteTemplate(String  id,
                                    boolean returnEntireResponse) throws PDFrePROException
    {
        // Send the request.
        AtomicInteger responseCode = new AtomicInteger();
        JsonValue     response     = this.sendRequest(PDFrePRO.URI_TEMPLATES_ID.replaceAll("\\{id\\}",
                                                                                           id),
                                                      "DELETE",
                                                      new JsonValue(new JsonObject()),
                                                      responseCode);
        
        // Check, whether the HTTP status code indicates no content.
        if(responseCode.intValue() == 204)
        {
            // Build a valid response body.
            JsonObject temp = new JsonObject();
            
            temp.put("code",
                     new JsonValue(204));
            temp.put("status",
                     new JsonValue("success"));
            temp.put("data",
                     new JsonValue(new JsonObject()));
            response.setValue(temp);
        }
        
        // Check, whether the entire response shall be returned.
        if(returnEntireResponse)
        {
            return response;
        }

        // Validate the response.
        this.validateResponse(response,
                              new int[]
                                  {
                                      204
                                  });
        
        // Return the result.
        return new JsonValue(response.getJsonObjectValue().get("status").getStringValue().equals("success"));
    } // End of method 'deleteTemplate'.

    //*********************************************************************************************
    //
    //                  Response Functions
    //
    //*********************************************************************************************

    /**
     * Validates a response, which were returned, due to a request to the PDFrePRO host.
     * 
     * @param JsonValue response - The response, which shall be validated.
     * 
     * @throws PDFrePROException - If the response is not valid or contains an error.
     */
    protected void validateResponse(JsonValue response) throws PDFrePROException
    {
        this.validateResponse(response,
                              new int[]
                                  {
                                      200
                                  });
    } // End of method 'validateResponse'.
    
    /**
     * Validates a response, which were returned, due to a request to the PDFrePRO host.
     * 
     * @param JsonValue response   - The response, which shall be validated.
     * @param int[]     validCodes - All valid HTTP status codes, which are expected for a success of this response.
     * 
     * @throws PDFrePROException - If the response is not valid or contains an error.
     */
    protected void validateResponse(JsonValue response,
                                    int[]     validCodes) throws PDFrePROException
    {
        // Check, whether the response is a JSON object.
        if(!response.isJsonObject())
        {
            throw new PDFrePROException("Response is invalid, due to invalid structure!");
        }
        
        // Get JSON object.
        JsonObject responseObject = response.getJsonObjectValue();
        
        // Check, whether code, status and data are available.
        if(!responseObject.containsKey("code") ||
           !responseObject.containsKey("status") ||
           !responseObject.containsKey("data"))
        {
            throw new PDFrePROException("Response is invalid, due to missing code, status or data!");
        }
        
        // Check, whether code and status are valid.
        if(!responseObject.get("code").isNumber() || !responseObject.get("status").isString())
        {
            throw new PDFrePROException("Response is invalid, due to invalid code or status!");
        }
        
        int code = responseObject.get("code").getNumberValue().intValue();
        
        if((!Arrays.stream(validCodes).anyMatch(x -> x == code) && !Arrays.stream(PDFrePRO.VALID_STATUS_CODES).anyMatch(x -> x == code)) ||
           !Arrays.stream(PDFrePRO.VALID_STATUSES).anyMatch(x -> x.equals(responseObject.get("status").getStringValue())))
        {
            throw new PDFrePROException("Response is invalid, due to invalid code or status!");
        }
        
        // Check, whether the response contains an error.
        if(responseObject.get("status").getValue().equals("success"))
        {
            // Check, whether data is a JSON object.
            if(!responseObject.get("data").isJsonObject())
            {
                throw new PDFrePROException("Response is invalid, due to invalid data!");
            }
        }
        else
        {
            // Check, whether message is available.
            if(!responseObject.containsKey("message"))
            {
                throw new PDFrePROException("Response is invalid, due to missing message!");
            }
            
            // Check, whether data and message are strings.
            if(!responseObject.get("data").isString() || !responseObject.get("message").isString())
            {
                throw new PDFrePROException("Response is invalid, due to invalid data or message!");
            }
            
            // Throw proper exception.
            throw new PDFrePROException(responseObject.get("data").getStringValue() + ": " + responseObject.get("message").getStringValue(),
                                        code);
        }
    } // End of method 'validateResponse'.

    //*********************************************************************************************
    //
    //                  Request Methods
    //
    //*********************************************************************************************

    /**
     * Sends a request.
     *
     * @param String resource - The resource, which shall be requested.
     *
     * @return JsonValue - The result of the request.
     *
     * @throws PDFrePROException - If the request could not be send, properly.
     */
    protected JsonValue sendRequest(String resource) throws PDFrePROException
    {
        return this.sendRequest(resource,
                                "GET",
                                new JsonValue(new JsonObject()),
                                new AtomicInteger());
    } // End of method 'sendRequest'.
    
    /**
     * Sends a request.
     *
     * @param String resource - The resource, which shall be requested.
     * @param String method   - The HTTP method for the request.
     *
     * @return JsonValue - The result of the request.
     *
     * @throws PDFrePROException - If the request could not be send, properly.
     */
    protected JsonValue sendRequest(String resource,
                                    String method) throws PDFrePROException
    {
        return this.sendRequest(resource,
                                method,
                                new JsonValue(new JsonObject()),
                                new AtomicInteger());
    } // End of method 'sendRequest'.
    
    /**
     * Sends a request.
     *
     * @param String  	resource - The resource, which shall be requested.
     * @param String  	method   - The HTTP method for the request.
     * @param JsonValue data     - The Data, which shall be send with the request.
     *
     * @return JsonValue - The result of the request.
     *
     * @throws PDFrePROException - If the request could not be send, properly.
     */
    protected JsonValue sendRequest(String 	  resource,
	    						    String 	  method,
	    						    JsonValue data) throws PDFrePROException
    {
        return this.sendRequest(resource,
                                method,
                                data,
                                new AtomicInteger());
    } // End of method 'sendRequest'.
    
    /**
     * Sends a request.
     *
     * @param String  		resource       - The resource, which shall be requested.
     * @param String  		method         - The HTTP method for the request.
     * @param JsonValue 	data           - The Data, which shall be send with the request.
     * @param AtomicInteger httpCode (out) - This parameter will set to hold the HTTP status code of the request.
     *
     * @return JsonValue - The result of the request.
     *
     * @throws PDFrePROException - If the request could not be send, properly.
     */
    protected JsonValue sendRequest(String 	      resource,
	    	 					    String 	      method,
	    	 					    JsonValue	  data,
	    						    AtomicInteger httpCode) throws PDFrePROException
    {
        try
        {
            // Create connection.
            URL               url        = new URL(this.host + resource);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();

            // Set request method.
            connection.setRequestMethod(method);
            
            // Initialize headers and data string, buffer and bytes array.
            String       accept        = "application/json;charset=utf-8";
            String       contentType   = "";
            String       contentLength = "";
            String       date          = ZonedDateTime.now(ZoneId.of("GMT")).format(DateTimeFormatter.RFC_1123_DATE_TIME);
            String       dataString    = "";
            StringBuffer buffer        = new StringBuffer();
            byte[]       bytes;
            
            // Do further stuff, depending on request method.
            if(method.equals("POST") || method.equals("PUT"))
            {
                // Set data string and headers.
                dataString    = data.toString();
                contentType   = "application/json;charset=utf-8";
                contentLength = String.valueOf(dataString.getBytes("UTF-8").length);
                
                // Calculate MD5 hash of data string.
                bytes = MessageDigest.getInstance("MD5").digest(dataString.getBytes("UTF-8"));
                
                for(int i = 0; i < bytes.length; i++)
                {
                    buffer.append(String.format("%02x",
                                                bytes[i] & 0xff));
                }
            }
            
            // Extract host and port of host-URL.
            String host = this.extractHost();
            
            // Prepare hash string for authorization header.
            String hashString  = this.apiKey                                             + "\\n";
                   hashString += method                                                  + "\\n";
                   hashString += resource                                                + "\\n";
                   hashString += host                                                    + "\\n";
                   hashString += (dataString.isEmpty() ? dataString : buffer.toString()) + "\\n";
                   hashString += accept                                                  + "\\n";
                   hashString += contentType                                             + "\\n";
                   hashString += contentLength                                           + "\\n";
                   hashString += date                                                    + "\\n";
                   
            // Calculate hash for authorization header.
                          buffer = new StringBuffer();
            Mac           mac    = Mac.getInstance("HmacSHA256");
            SecretKeySpec key    = new SecretKeySpec(this.sharedKey.getBytes("UTF-8"),
                                                     "HmacSHA256");
            
            mac.init(key);
            
            bytes = mac.doFinal(hashString.getBytes("UTF-8"));
            
            for(int i = 0; i < bytes.length; i++)
            {
                buffer.append(String.format("%02x",
                                            bytes[i] & 0xff));
            }
            
            // Set headers.
            connection.setRequestProperty("Accept",
                                          accept);
            connection.setRequestProperty("Date",
                                          date);
            connection.setRequestProperty("Authorization",
                                          "SharedKey " + this.apiKey + ":" + buffer.toString());

            // Do further stuff, depending request method.
            if(method.equals("POST") || method.equals("PUT"))
            {
                // Set content headers.
                connection.setRequestProperty("Content-Type",
                                              contentType);
                connection.setRequestProperty("Content-Length",
                                              contentLength);
                
                // Send content.
                connection.setDoOutput(true);
                
                DataOutputStream dos = new DataOutputStream(connection.getOutputStream());
                
                dos.write(dataString.getBytes("UTF-8"));
                dos.flush();
                dos.close();
            }
            
            // Get response code.
            httpCode.set(connection.getResponseCode());
            
            // Get response.
            boolean        success   = !Arrays.stream(PDFrePRO.VALID_STATUS_CODES).anyMatch(x -> x == httpCode.intValue());
            BufferedReader br        = new BufferedReader(new InputStreamReader(success ? connection.getInputStream() : connection.getErrorStream()));
                           buffer    = new StringBuffer();
            String         inputLine;
            
            while((inputLine = br.readLine()) != null)
            {
                buffer.append(inputLine);
            }
            
            br.close();
            
            // Return response.
            return Json.JsonDecode(buffer.toString());
        }
        catch(Exception exception)
        {
            throw new PDFrePROException("Request could not be send, properly!",
                                        exception);
        }
    } // End of method 'sendRequest'.

    //*********************************************************************************************
    //
    //                  Helper Functions
    //
    //*********************************************************************************************

    /**
     * Extracts the host and port out of the stored host address of this class instance and returns it back as
     * concatenated string.
     *
     * @return String - The extracted and concatenated host and port of the stored host address of this class instance.
     * 
     * @throws MalformedURLException - If the host address could not be parsed into an URL.
     */
    private String extractHost() throws MalformedURLException
    {
        URL    url  = new URL(this.host);
        String host = url.getHost();
        int    port = url.getPort();
        
        if(port != -1)
        {
            host += ':' + String.valueOf(port);
        }
        
        return host;
    } // End of method 'extractHost'.
}
