
//*************************************************************************************************
//
//                      Package 'json'
//
//*************************************************************************************************

package json;

//*************************************************************************************************
//
//                      Imports
//
//*************************************************************************************************

// Package 'java.util'.
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

// Package 'javax.lang.model.type'.
import javax.lang.model.type.NullType;

//*************************************************************************************************
//
//                      Class 'Json'
//
//*************************************************************************************************

/**
 * This class is responsible for json en- and decoding.
 */
public final class Json
{
	
	//*********************************************************************************************
	//
	//                  Constructors
	//
	//*********************************************************************************************

	/**
	 * Private Json constructor, because this class should never be instantiated.
	 */
	private Json()
	{
	} // End of constructor.

	//*********************************************************************************************
	//
	//                  JSON Methods
	//
	//*********************************************************************************************

	/**
	 * Encodes 'null' into a JSON string.
	 * 
	 * @param NullType value - The 'null' value, which shall be encoded.
	 * 
	 * @return A JSON string, containing 'null'.
	 */
	public static String JsonEncode(NullType value)
	{
		return "null";
	} // End of method 'JsonEncode'.

	/**
	 * Encodes a boolean into a JSON string.
	 * 
	 * @param Boolean value - The boolean, which shall be encoded.
	 * 
	 * @return A JSON string, containing the boolean.
	 */
	public static String JsonEncode(Boolean value)
	{
		return value.toString();
	} // End of method 'JsonEncode'.

	/**
	 * Encodes a number into a JSON string.
	 * 
	 * @param Number value - The number, which shall be encoded.
	 * 
	 * @return A JSON string, containing the number.
	 */
	public static String JsonEncode(Number value)
	{
		return value.toString();
	} // End of method 'JsonEncode'.

	/**
	 * Encodes a string into a JSON string.
	 * 
	 * @param String value - The string, which shall be encoded.
	 * 
	 * @return A JSON string, containing the string.
	 */
	public static String JsonEncode(String value)
	{
		return "\"" + value.replaceAll("\\\\{0}\\\"",
		                               "\\\\\\\"") + "\"";
	} // End of method 'JsonEncode'.

	/**
	 * Encodes a JSON array into a JSON string.
	 * 
	 * @param JsonArray value - The JSON array, which shall be encoded.
	 * 
	 * @return A JSON string, containing the JSON array.
	 */
	public static String JsonEncode(JsonArray value)
	{
		return value.toString();
	} // End of method 'JsonEncode'.
	
	/**
	 * Encodes a JSON object into a JSON string.
	 * 
	 * @param JsonObject value - The JSON object, which shall be encoded.
	 * 
	 * @return A JSON string, containing the JSON object.
	 */
	public static String JsonEncode(JsonObject value)
	{
		return value.toString();
	} // End of method 'JsonEncode'.

	/**
	 * Encodes a JSON value into a JSON string.
	 * 
	 * @param JsonValue value - The JSON value, which shall be encoded.
	 * 
	 * @return A JSON string, containing the JSON value.
	 */
	public static String JsonEncode(JsonValue value)
	{
		return value.toString();
	} // End of method 'JsonEncode'.

	/**
	 * Decodes a JSON string into a JSON value.
	 * 
	 * @param String json - The JSON string, which shall be decoded.
	 * 
	 * @return The decoded JSON string.
	 */
	public static JsonValue JsonDecode(String json)
	{
	    // Check, whether the JSON string is empty.
	    if((json == null) || (json.isEmpty()))
	    {
	        return new JsonValue(new JsonObject());
	    }
	    
		// Check, whether the JSON string contains a JSON object.
		if(json.startsWith("{") && json.endsWith("}"))
		{
			return new JsonValue(Json.DecodeObject(json));
		}
		
		// Check, whether the JSON string contains a JSON array.
		if(json.startsWith("[") && json.endsWith("]"))
		{
			return new JsonValue(Json.DecodeArray(json));
		}

		// Check, whether the JSON string contains a string.
		if(json.startsWith("\"") && json.endsWith("\""))
		{
			return new JsonValue(Json.DecodeString(json));
		}

		// Check, whether the JSON string contains 'null'.
		if(json.equals("null"))
		{
			return new JsonValue();
		}
		
		// Check, whether the JSON string contains a boolean.
		if(json.equals(Boolean.FALSE.toString()) || json.equals(Boolean.TRUE.toString()))
		{
			return new JsonValue(Boolean.valueOf(json));
		}
		
		// Check, whether the JSON string contains an integer.
		if(json.matches("^[-+]?[0-9]+$"))
		{
			return new JsonValue(Integer.valueOf(json));
		}
		
		// The JSON string seems to contain a float.
		return new JsonValue(Float.valueOf(json));
	} // End of method 'JsonDecode'.

    /**
     * Decodes a JSON string into a string.
     * 
     * @param String json - The JSON string, which shall be decoded.
     * 
     * @return The decoded JSON string.
     */
    private static String DecodeString(String json)
    {
        StringBuffer result = new StringBuffer();
        
        // Remove quote chars.
        json = json.substring(1,
                              json.length() - 1);
        
        // Decode encoded chars, except they are still in a JSON string.
        char[] string    = json.toCharArray();
        int    lastIndex = 0;
        int    depth     = 0;
        
        for(int i = 0; i < string.length; i++)
        {
            // Decode chars '"' and '/'.
            if((depth <= 0) && (string[i] == '\\') && "\"/".contains("" + string[i + 1])) {
                result.append(Arrays.copyOfRange(string, lastIndex, i));

                lastIndex = i + 1;
            }

            // Set depth.
            if((string[i] == '"') && (string[i - 1] != '\\'))
            {
                depth++;
                
                continue;
            }
            if((string[i] == '"') && (string[i - 1] != '\\'))
            {
                depth--;
                
                continue;
            }
        }

        // Add last element.
        result.append(Arrays.copyOfRange(string, lastIndex, string.length));
        
        // Return elements.
        return result.toString();
    } // End of method 'DecodeString'.
    
	/**
	 * Decodes a JSON string into a JSON array.
	 * 
	 * @param String json - The JSON string, which shall be decoded.
	 * 
	 * @return The decoded JSON string.
	 */
	private static JsonArray DecodeArray(String json)
	{
		JsonArray jsonArray = new JsonArray();
		
		// Remove brackets.
		json = json.substring(1,
							  json.length() - 1);
		
		// Split the string up into its elements.
		String[] elements = Json.splitJson(json);
		
		// For each element...
		for(String element : elements)
		{
			// ...store it into the JSON array.
			jsonArray.add(Json.JsonDecode(element));
		}
		
		return jsonArray;
	} // End of method 'DecodeArray'.
	
	/**
	 * Decodes a JSON string into a JSON object.
	 * 
	 * @param String json - The JSON string, which shall be decoded.
	 * 
	 * @return The decoded JSON string.
	 */
	private static JsonObject DecodeObject(String json)
	{
		JsonObject jsonObject = new JsonObject();
		
		// Remove brackets.
		json = json.substring(1,
							  json.length() - 1);
		
		// Split the string up into its elements.
		String[] elements = Json.splitJson(json);
		
		// For each element...
		for(String element : elements)
		{
			// ...split it up into key and value.
			String[] pair = element.split(":",
			                              2);
			
			// ...store it into the JSON object.
			jsonObject.put(pair[0].substring(1,
											 pair[0].length() - 1),
						   Json.JsonDecode(pair[1]));
		}
		
		return jsonObject;
	} // End of method 'DecodeObject'.

    //*********************************************************************************************
    //
    //                  Helper Methods
    //
    //*********************************************************************************************

	/**
	 * Splits a JSON string of an array or object into its elements.
	 * 
	 * @param String json - The JSON string, which shall be splitted.
	 * 
	 * @return String[] - The single elements of the array or object.
	 */
	private static String[] splitJson(String json)
	{
	    List<String> elements     = new ArrayList<String>();
        char[]       string       = json.toCharArray();
        boolean      insideString = false;
        int          depth        = 0;
        int          lastIndex    = 0;
	    
        // Iterate through JSON string.
	    for(int i = 0; i < string.length; i++)
	    {
	        // Cut, if necessary.
	        if(!insideString && (depth <= 0) && (string[i] == ','))
	        {
	            elements.add(new String(Arrays.copyOfRange(string,
	                                                       lastIndex,
	                                                       i)));
	            
	            lastIndex = i + 1;
	            
	            continue;
	        }
	        
	        // Check for JSON string.
	        if(!insideString && (string[i] == '"'))
	        {
	            insideString = true;
	            
	            continue;
	        }
	        if(insideString)
	        {
	            if(string[i] == '"')
	            {
	                insideString = false;
	            }
	            
	            continue;
	        }
	        
	        // Set depth.
	        if((string[i] == '{') || (string[i] == '['))
	        {
	            depth++;
	            
	            continue;
	        }
	        if((string[i] == '}') || (string[i] == ']'))
	        {
	            depth--;
	            
	            continue;
	        }
	    }
	    
	    // Add last element.
	    elements.add(new String(Arrays.copyOfRange(string,
	                                               lastIndex,
	                                               string.length)));
	    
	    // Return elements.
	    return elements.toArray(new String[elements.size()]);
	} // End of method 'splitJson'.
	
} // End of class 'Json'.
