
//*************************************************************************************************
//
// PDFrePRO - C# example class for how to print a PDF of one of your templates!
//
//*************************************************************************************************

/* Tested with .NET versions 4.5, 4.5.1, 4.5.2, 4.6, 4.6.1, 4.7, 4.7.1 and 4.7.2. */
/* Tested with Newtonsof.Json versions 10.0.3 and 12.0.3. */

// Importing the Newtonsoft JSON classes is recommended, because they are used by this example.
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

// These imports are necessary for printing a PDF of your template.
using System;
using System.Diagnostics;
using System.IO;
using System.Net.Http;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace PDFrePRO {

    public class Print_PDF {

        // The following settings should you copy from your account in the PDFrePRO portal:

        // One of your personal API keys, which normally represents one of your applications.
        public static string apiKey     = "<your-api-key>";

        // The associated shared key to your chosen API key.
        public static string sharedKey  = "<your-shared-key>";

        // One of your prepared templates, for which you want to print a PDF.
        public static string templateId = "<your-template-ID>";

        public static void Main(string[] args) {
            try {
                // Send the request to PDFrePRO-API and get the response.
                JObject response = SendRequest("/v3/templates/" + templateId + "/pdf", MyData()).Result;

                // Get the PDF.
                string pdf = response.GetValue("data").Value<string>("pdf");

                // Store the PDF as file on your desktop.
                Byte[]     pdfContent = Convert.FromBase64String(pdf);
                string     filePath   = Environment.GetFolderPath(Environment.SpecialFolder.Desktop) + Path.DirectorySeparatorChar + "my_first_print_result.pdf";
                FileStream fs         = File.Create(filePath);

                fs.Write(pdfContent, 0, pdfContent.Length);
                fs.Close();

                // Open the PDF file, which got just created.
                Process.Start(filePath);
            } catch (Exception exception) {
                // Print any occurring exception.
                Console.WriteLine("<pre>" + exception.ToString() + "</pre>");
            }
        }

        // The following structure "myData" represents the data array you want to merge into your PDF document.
        // It's normally populated based on your users input or a SQL result.
        //
        // The data to print in this example is located in a JSON file which we now read in directly from the
        // filesystem:
        public static JObject MyData() {
            // Get content of JSON file.
            StreamReader   st   = File.OpenText("my_example_data_to_print.json");
            JsonTextReader jtr  = new JsonTextReader(st);
            JObject        json = (JObject)JToken.ReadFrom(jtr);

            // Close streams.
            jtr.Close();
            st.Close();

            // Return decoded JSON.
            return json;
        }

        // This is the host address of PDFrePRO-API.
        private static string host = "https://api.pdfrepro.de";

        // This method sends a request to the API of PDFrePRO and returns the response.
        private static async Task<JObject> SendRequest(string  resource, JObject data) {
            try {
                // Prepare JSON data.
                JObject temp = new JObject();

                temp.Add("data", data);

                data = temp;

                // Initialize an HTTP client and convert the JSON data into a JSON string.
                Uri        uri        = new Uri(host + resource);
                HttpClient client     = new HttpClient();
                string     dataString = data.ToString(Formatting.None);

                // Initialize headers.
                string accept      = "application/json;charset=utf-8";
                string contentType = "application/json;charset=utf-8";
                string date        = DateTime.UtcNow.ToString("ddd, dd MMM yyyy HH:mm:ss", new System.Globalization.CultureInfo("en-US")) + " GMT";

                // Prepare content-hash for authorization header.
                MD5    md5         = MD5.Create();
                string contentHash = BitConverter.ToString(md5.ComputeHash(Encoding.UTF8.GetBytes(dataString))).Replace("-", "").ToLower();

                // Prepare hash string for authorization header.
                string hashString  = apiKey                                                     + "\\n";
                       hashString += "POST"                                                     + "\\n";
                       hashString += resource                                                   + "\\n";
                       hashString += (uri.IsDefaultPort ? uri.Host : uri.Host + ":" + uri.Port) + "\\n";
                       hashString += contentHash                                                + "\\n";
                       hashString += accept                                                     + "\\n";
                       hashString += contentType                                                + "\\n";
                       hashString += Encoding.UTF8.GetByteCount(dataString).ToString()          + "\\n";
                       hashString += date                                                       + "\\n";

                // Calculate hash for authorization header.
                HMACSHA256 hmac = new HMACSHA256(Encoding.UTF8.GetBytes(sharedKey));
                string     hash = BitConverter.ToString(hmac.ComputeHash(Encoding.UTF8.GetBytes(hashString))).Replace("-", "").ToLower();

                hmac.Dispose();

                // Set headers.
                client.DefaultRequestHeaders.Add("Accept",        accept);
                client.DefaultRequestHeaders.Add("Authorization", "SharedKey " + apiKey + ":" + hash);
                client.DefaultRequestHeaders.Add("Date",          date);

                // Get response message.
                HttpResponseMessage responseMessage = await client.PostAsync(uri, new StringContent(dataString, Encoding.UTF8, "application/json"));

                // Get response content.
                string responseContent = await responseMessage.Content.ReadAsStringAsync();

                // Return response.
                return await Task.Run(() => JObject.Parse(responseContent));
            } catch (Exception exception) {
                throw new Exception("Request could not be send, properly!", exception);
            }
        }

    }

}
